C
C  lib_tirem.f
C
C  ***
C
C  1.2.1 020927 rsj
C  Ported to OS X, only the version number changed.
C
C  1.2bb 980323 rsj
C  Ported to Linux, trivial changes.
C
C  1.2 971111 rsj
C  Added "automatic" mode to EPM-73 modifications.
C
C  1.1 950222 rsj
C  Created module.
C
C  ***
C
        SUBROUTINE TIREM(H,NPT,PSP,TXH,RXH,FRQ,IHV,IEP,XNS,SIG,PER,
     &     FSLO,PLOS,IMOD)
C
C  THIS IS A "BLACK-BOX" SUBROUTINE FORM OF THE TIREM ROUTINE, ADAPTED
C  FROM THE ONE ORIGINALLY CONTAINED IN THE propg PROGRAM.  ONLY MINOR
C  CHANGES WERE NECESSARY INVOLVING THE PASSING OF ARGUMENTS.  MANY OF
C  THE ARGUMENTS ARE IMMEDIATELY MOVED INTO COMMON; IN propg THESE WERE
C  NOT ARGUMENTS BUT WERE PLACED DIRECTLY INTO COMMON BY OTHER CODE
C  BEFORE TIREM() WAS CALLED.  THE PASSED ARGUMENTS ARE:
C
C    H(*) - ARRAY OF PROFILE ELEVATIONS IN FEET
C    NPT  - INDEX TO RECEIVER POINT, DEFINES PATH LENGTH
C    PSP  - PROFILE POINT SPACING IN MILES
C    TXH  - TRANSMITTER HEIGHT IN FEET AMSL
C    RXH  - RECEIVER HEIGHT IN FEET AGL
C    FRQ  - FREQUENCY IN MHZ
C    IHV  - POLARIZATION, 0 = H, 1 = V
C    IEP  - 0 = NORMAL TIREM, 1 = MODIFIED TIREM USING EPM-73 MODES,
C           2 = AUTOMATIC SELECTION OF NORMAL TIREM OR EPM-73 TIREM
C    XNS  - ATMOSPHERIC REFRACTIVITY IN N-UNITS
C    SIG  - GROUND CONDUCTIVITY IN S/M
C    PER  - GROUND PERMITTIVITY
C
C  NOTE THAT ANY OF THE LAST THREE ARGUMENTS CAN BE PASSED AS 0, IN WHICH
C  CASE A STANDARD DEFAULT VALUE WILL BE USED FOR THAT PARAMETER.
C
C  THE RETURNED ARGUMENTS ARE:
C
C    FSLO - PATH LOSS IN DB BASED ON FREE-SPACE
C    PLOS - PATH LOSS IN DB BASED ON TERRAIN-SENSITIVE MODE
C    IMOD - TIREM MODEL NUMBER USED, 0 = ERROR
C
C  THE TERRAIN-INTEGRATED ROUGH-EARTH MODEL ( TIREM ) COMPUTES THE
C  MEDIAN PROPAGATION LOSS OVER A GIVEN POINT-TO-POINT PATH. TIREM
C  EXAMINES THE GEOMETRY OF THE GIVEN TERRAIN PROFILE AND THEN BRANCHES
C  TO THE APPROPRIATE SUB-MODULE(S). TYPICALLY, THE PATH LOSS VALUE
C  RETURNED BY TIREM IS A WEIGHTED COMBINATION OF LOSSES FROM
C  SEVERAL SUB-MODULES.
C
C  THIS FORMULATION OF TIREM IS BASED ON THE PROPAGATION MODEL OF THE
C  SAME NAME USED AT THE ELECTROMAGNETIC COMPATIBILITY ANALYSIS
C  CENTER ( ECAC ).
C
C  FOR H&E PURPOSES, THE TROPOSPHERIC FORWARD SCATTER MODEL IS NOT VERY
C  USEFUL (PATHS WE STUDY ARE NOT THAT LONG), BESIDES WE ARE MISSING A
C  NECESSARY DATA FILE FULL OF TABLES FOR THAT MODEL. HENCE, THE MODULES
C  TROSCP AND R50 HAVE BEEN ELIMINATED.
C
C  THE MAJOR SUB-MODULES OF TIREM ARE :
C         ALONG   -   LINE-OF-SIGHT MODEL
C         KED     -   KNIFE-EDGE AND EFFECTIVE KNIFE-EDGE MODEL
C         RED     -   ROUGH-EARTH BEYOND L-O-S MODEL
C         TROSCP  -   TROPOSPHERIC FORWARD SCATTER MODEL
C         R50     -   REFINEMENT OF TROPO PATH LOSS PREDICTION
C
C  PRIMARY MODELS (INITIAL MODE SELECTION BASED ON PATH GEOMETRY) :
C         1   -   BEYOND L-O-S, EFFECTIVE KNIFE-EDGE  (KED,TROSCP,R50)
C         2   -   BEYOND L-O-S, SINGLE KNIFE-EDGE  (KED,TROSCP,R50)
C         4   -   BEYOND L-O-S, NEAR KNIFE-EDGE (KED,RED,TROSCP,R50)
C         6   -   BEYOND L-O-S, WIDELY SEPARATED HORIZONS (RED,TROSCP,R50)
C         7   -   LINE-OF-SIGHT   (ALONG)
C
C  SECONDARY MODELS (REFINEMENTS,MODIFICATIONS TO PRIMARY MODES)
C         5   -   L-O-S, MARGINAL FRESNEL ZONE INTERSECTION (ALONG)
C         8   -   L-O-S, FREE-SPACE PATH (ALONG)
C
C  ADDITIONAL MODES IN H&E VERSION WITH EPM-73 ACTIVE:
C         3   -   MODE 5 WITH EPM-73 SUBSTITUTED FOR LONGLEY-RICE
C         9   -   MODE 7 WITH EPM-73 SUBSTITUTED FOR LONGLEY-RICE
C
C  NOTE: THE FOLLOWING MODES ARE NOT FUNCTIONAL IN THIS MODIFIED H&E VERSION,
C  DESCRIPTIONS ARE LEFT FOR REFERENCE.  NOTE THAT THE NUMBER 9 IS RE-USED
C  FOR ONE OF THE EPM-73 MODES, SEE ABOVE.
C
C         9   -   BEYOND L-O-S, TROPO PATH (...R50)
C        10   -   BEYOND L-O-S, MODIFIED NON-TROPO (...R50)
C        11   -   BEYOND L-O-S, WEIGHTED COMBINATION WITH EFFECTIVE
C                 DOUBLE KNIFE-EDGE LOSS (...R50)
C        12   -   BEYOND L-O-S, EFFECTIVE DOUBLE KNIFE-EDGE LOSS (...R50)
C        13   -   BEYOND L-O-S, WEIGHTED COMBINATION WITH EFFECTIVE
C                 DOUBLE KNIFE-EDGE LOSS (...R50)
C
C  THE VARIABLES LISTED BELOW ARE MOSTLY LOCATED IN COMMON BLOCKS; MOST OF THE
C  ARGUMENTS TO THIS SUBROUTINE VERSION OF TIREM ARE IMMEDIATELY COPIED TO THE
C  APPROPRIATE COMMON VARIABLES, AND VICE-VERSA FOR RETURNS.
C
C            INPUT
C              FREQ   -  FREQUENCY (MHZ)
C              XN     -  REFRACTIVITY
C              SIGMA  -  CONDUCTIVITY (MHO/M)
C              EPS    -  PERMITTIVITY
C              RE     -  EARTH RADIUS
C              HST    -  TX STRUCTURAL ANTENNA HEIGHT
C              HSR    -  RX STRUCTURAL ANTENNA HEIGHT
C              DST    -  PATH DISTANCE
C              NELEV  -  NUMBER OF POINTS IN PROFILE
C              DINC   -  PROFILE DISTANCE INCREMENT
C              H      -  ARRAY OF PROFILE ELEVATIONS
C
C           OUTPUT
C              PATLOS -  PATH LOSS
C              MODEL  -  PATH MODEL
C
C           OTHER MAJOR VARIABLES
C              PLFS   -  FREE SPACE LOSS
C              PLKED  -  KNIFE-EDGE LOSS
C              PLRED  -  ROUGH-EARTH LOSS
C              PLONG  -  L-O-S LOSS
C              PLTRO  -  TROPO LOSS
C              PLR50  -  REFINED (BEYOND L-O-S) LOSS
C              HTS    -  TX ANTENNA ELEVATION
C              HRS    -  RX ANTENNA ELEVATION
C              HTE    -  TX EFFECTIVE ANT HGT
C              HRE    -  RX EFFECTIVE ANT HGT
C              LT     -  TX HORIZON POINT
C              LR     -  RX HORIZON POINT
C              DLT    -  DIST TO TX HORIZON
C              DLR    -  DIST TO RX HORIZON
C              ALPH00 -  ANGLE BETWEEN ANT-TO-ANT RAY AND TX HORIZON
C              BETA00 -  ANGLE BETWEEN ANT-TO-ANT RAY AND RX HORIZON
C              THET00 -  PATH ANGULAR DISTANCE ( =ALPH00+BETA00 )
C              HUMID  -  HUMIDITY (PRESENTLY NOT USED)
C              IPOL   -  POLARIZATION
C              AK     -  EARTH RADIUS CORRECTION FACTOR
C
C          OTHER SUBROUTINES CALLED
C              EXTRA  -  EXTRACTS GEOMETRIC INFORMATION FROM PROFILE
C              PMS    -  MAKES INITIAL MODEL SELECTION
C              FSLOSS -  COMPUTES FREE SPACE PATH LOSS
C
        REAL H(*)
C
C  THE FOLLOWING PARAMETER SETS AN UPPER LIMIT ON THE NUMBER OF POINTS IN THE
C  PROFILE.  THIS IS NEEDED BECAUSE OF THE ALONG() MODULE, WHICH INTERNALLY
C  USES AN ARRAY THAT MUST HOLD 1/5TH OF THE NUMBER OF POINTS IN THE ACTUAL
C  PROFILE.  TO AVOID HAVING TO MAKE MAJOR CHANGES IN ALONG() SO THAT ARRAY
C  COULD BE DYNAMICALLY ALLOCATED, WE SIMPLY SET AN UPPER LIMIT.  THE PARAMETER
C  MAXD DEFINED IN MODULE ALONG() MUST BE KEPT IN SYNC IF THIS IS CHANGED; SEE
C  COMMENTS THERE.
C
        PARAMETER (NELMAX=5000)
        COMMON /LOSS/MODEL,PATLOS,PLFS,PLKED,PLRED,PLONG
        COMMON /PROFIL/NELEV,DINC
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
C
C  A COMMON BLOCK OF CONSTANTS USED THROUGHOUT.
C
        COMMON /CONST/FSM2FT,FM2FT,FKM2SM,FNM2SM,FR2D,PI
        DATA FSM2FT/5280./FNM2SM/1.150758/FM2FT/3.280840/
     &       FKM2SM/.6213712/FR2D/57.2957795/PI/3.1415926/
C
C  CHECK THE PROFILE LENGTH AGAINST THE MAX LIMIT.
C
        IF (NPT.GT.NELMAX) GOTO 120
C
C  DO ARGUMENT PROCESSING - SOME CALCULATIONS AND DERIVATIONS THAT USED TO BE
C  DONE BY THE MAIN ROUTINE IN propg.
C
        NELEV=NPT
        DINC=PSP
        DST=REAL(NPT-1)*PSP
        HST=TXH-H(1)
        HSR=RXH
        FREQ=FRQ
        IPOL=IHV
C
C  THE EPM-73 KEY VALUE 2 MEANS THAT USE OF EPM-73 IN PLACE OF LONGLEY-RICE IN
C  THE APPROPRIATE MODES IS SELECTED AUTOMATICALLY.  TO START WITH, IF THE
C  TRANSMITTER FREQUENCY IS OUT OF THE VALID RANGE FOR EPM-73, THEN EPM-73
C  WILL NEVER BE USED, SET THE INTERNAL FLAG TO 0.  IF THE INTERNAL FLAG KEEPS
C  THE VALUE 2, THEN FURTHER DECISIONS WILL BE MADE LATER WHETHER TO USE EPM-73
C  OR LONGLEY-RICE BASED ON A COMPARISON OF THE LOSS PREDICTED BY EACH.
C
        KEPM=IEP
        IF (KEPM.EQ.2) THEN
           IF (FREQ.GT.220.) THEN
              KEPM=0
           ENDIF
        ENDIF
C
C  THE THREE CONSTANTS THAT AFFECT TIREM, XNS, SIG, AND PER, HAVE DEFAULT
C  VALUES IF 0.'S ARE PASSED.  A COUPLE OF OTHER PARAMETERS ARE COMPUTED.
C
        IF (XNS.EQ.0.) THEN
           XN=310.
        ELSE
           XN=XNS
        ENDIF
        IF (SIG.EQ.0.) THEN
           SIGMA=0.0278
        ELSE
           SIGMA=SIG
        ENDIF
        IF (PER.EQ.0.) THEN
           EPS=15.
        ELSE
           EPS=PER
        ENDIF
        RE=3960.
        AK=1.0/(1.0-((.04663*EXP(.005577*XN))/(1.0+(XN*1.0E-6))))
C
C  BEGIN TIREM CALCULATIONS.  EXTRACT GEOMETRICAL PARAMETERS FROM PROFILE.
C
        CALL EXTRA(H)
C
C  MAKE INITIAL MODEL SELECTION.
C
        CALL PMS
C
C  COMPUTE FREE SPACE LOSS.
C
        CALL FSLOSS
C
C  BRANCH TO APPROPRIATE SUB-MODULES AND COMPUTE PATH LOSS.
C
        IF ((MODEL.LT.1).OR.(MODEL.GT.7)) GOTO 120
        GOTO (20,20,120,90,120,30,50) MODEL
C
C  MODELS 1 AND 2.
C
 20     CALL KED(H)
        TEMP1=PLKED
        GOTO 100
C
C  MODEL 6.
C
 30     CALL RED
        TEMP1=PLRED
        GOTO 100
C
C  MODEL 7 (ENCOMPASSING MODELS 5,7,8) (AND 3,9 WITH EPM-73 ACTIVE).
C
 50     CALL ALONG(H)
        TEMP1=PLONG
        GOTO 100
C
C  MODEL 4.
C
 90     CALL KED(H)
        CALL RED
        F=(DST-DLT-DLR)/2.
        TEMP1=(1.0-F)*PLKED+F*PLRED
        GOTO 100
C
C  FOR ALL MODELS, CHECK RESULT AND RETURN.
C
 100    PATLOS=TEMP1
        IF (PATLOS.LE.1.0E-5) GOTO 120
C
C  NORMAL CONCLUSION, PUT COMMON VALUES INTO ARGUMENTS AND RETURN.
C
        FSLO=PLFS
        PLOS=PATLOS
        IMOD=MODEL
        RETURN
C
C  AN ERROR OCCURRED, RETURN ERROR CONDITION.
C
 120    FSLO=0.
        PLOS=0.
        IMOD=0
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE EXTRA(H)
C
C  EXTRA EXAMINES THE TERRAIN PROFILE AND EXTRACTS THE
C  INFORMATION REQUIRED FOR INITIAL MODEL SELECTION.
C
C  AN ITERATIVE PROCESS IS USED TO COMPUTE THE RADIO HORIZONS,
C  AND THE EFFECTIVE ANTENNA HEIGHTS : THE FIRST ITERATION USES
C  HTS AND HRS AS THE STRUCTURAL HEIGHTS + GROUND ELEVATIONS OF
C  THE END POINTS ; THE SECOND ITERATION USES HTS AND HRS CORRECTED
C  BY THE NBS EXPONENTIAL REFERENCE ATMOSPHERE MODEL.
C
C      INPUT
C
C          NELEV   -  NO OF ELEVS IN PROFILE
C          DINC    -  PROFILE DISTANCE INCREMENT
C          H       -  PROFILE ELEVATIONS (MSL)
C          DST     -  LENGTH OF PROFILE
C          HST     -  TX STRUCTURAL ANT HGT
C          HSR     -  RX STRUCTURAL ANT HGT
C
C      OUTPUT
C          HTS     -  CORRECTED TX ANT ELEV
C          HRS     -  CORRECTED RX ANT ELEV
C          HTE     -  EFFECTIVE TX ANT HGT
C          HRE     -  EFFECTIVE RX ANT HGT
C          LT      -  HORIZON POINT FROM TX
C          LR      -  HORIZON POINT FROM RX
C          LT10    -  TX TEN-MILE PT
C          LR10    -  RX TEN-MILE PT
C          DLT     -  TX RADIO HORIZON DIST
C          DLR     -  RX RADIO HORIZON DIST
C          ALPH00  -  TX COMPONENT ANGLE
C          BETA00  -  RX COMPONENT ANGLE
C          THET00  -  PATH ANGULAR DISTANCE ( = ALPH00+BETA00 )
C          RDA     -  CORRECTION FACTOR ( = 2*RE*AK )
C          DRDA    -  CORRECTION FACTOR ( = DST**2/RDA )
C
C      SUBROUTINES CALLED
C          HORIZ   -  COMPUTES RADIO HORIZONS
C          EFFHGT  -  COMPUTES EFFECTIVE ANTENNA HEIGHTS
C          TRAYTR   -  COMPUTES NBS RAY TRACE CORRECTION
C          ANGDST  -  COMPUTES ALPH00, BETA00, THET00
C
        REAL H(*)
        COMMON /LOSS/MODEL,PATLOS,PLFS,PLKED,PLRED,PLONG
        COMMON /PROFIL/NELEV,DINC
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
        COMMON /GEOM2/LT10,LR10,RDA,DRDA
C
C  CALC VALUES FOR HORIZONS,EFFECTIVE HEIGHTS
C
        RDA=RE*AK*2.
        DRDA=(DST**2)/RDA
C
C  LOCATE TX AND RX TEN-MILE POINTS
C
        LT10=NELEV
        LR10=1
        IF (DST.LE.10.) GOTO 20
        LT10=10./DINC
        LR10=NELEV-LT10
C
C  ADD GROUND HEIGHTS TO STRUCTURE HEIGHTS FOR FIRST TRY
C
 20     HTS=HST+H(1)
        HRS=HSR+H(NELEV)
        CALL HORIZ(H)
        IF (LR.EQ.LT) GOTO 40
        CALL EFFHGT(H)
        TEMP1=HTS
        TEMP2=HRS
        CALL TRAYTR(HTE,HTS)
        CALL TRAYTR(HRE,HRS)
        IF (ABS(HTS-TEMP1).LT.1.) GOTO 40
        IF (ABS(HRS-TEMP2).LT.1.) GOTO 40
C
C  SECOND ITERATION FOR NON-KNIFE-EDGE PATH
C
        CALL HORIZ(H)
        IF (LR.EQ.LT) GOTO 50
        CALL EFFHGT(H)
C
C  CALC VERT ANGLES
C
 40     CALL ANGDST(H)
        IF (LT.NE.LR) GOTO 80
C
C  SECOND ITERATION FOR SINGLE KNIFE-EDGE PATH
C
 50     IF (THET00.GT.0.) GOTO 60
        LT=NELEV
        HLT=H(LT)
        LR=1
        HLR=H(LR)
        DLT=DST
        DLR=DST
 60     CALL EFFHGT(H)
        TEMP1=HTS
        TEMP2=HRS
        CALL TRAYTR(HTE,HTS)
        CALL TRAYTR(HRE,HRS)
        IF (ABS(HTS-TEMP1).LT.1.) GOTO 70
        IF (ABS(HRS-TEMP2).LT.1.) GOTO 70
        CALL HORIZ(H)
        IF (LT.EQ.LR) GOTO 70
        CALL EFFHGT(H)
 70     CALL ANGDST(H)
C
C  ALL DONE
C
 80     CONTINUE
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE HORIZ(H)
C
C  HORIZ COMPUTES THE RADIO HORIZON DISTANCE FROM TX AND FROM RX.
C
C     INPUT
C        NELEV   -  NO OF PROFILE PTS
C        DINC    -  PROFILE INCREMENT
C        H       -  PROFILE ELEVATIONS (FT)
C        DST     -  PROFILE DISTANCE
C        HST     -  TX ANT ELEV
C        HSR     -  RX ANT ELEV
C        RDA     -  ELEVATION CORRECTION FACTOR
C        DRDA    -  ELEVATION CORRECTION FACTOR ( DST**2/RDA )
C        FSM2FT  -  SM-TO-FT CONVERSION FACTOR
C
C     OUTPUT
C        LT      -  HORIZON PT FROM TX
C        LR      -  HORIZON PT FROM RX
C        DLT     -  DISTANCE FROM TX TO HORIZON
C        DLR     -  DISTANCE FROM RX TO HORIZON
C
        REAL H(*)
        COMMON /PROFIL/NELEV,DINC
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /GEOM2/LT10,LR10,RDA,DRDA
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
        COMMON /CONST/FSM2FT,FM2FT,FKM2SM,FNM2SM,FR2D,PI
C
C  INITIALIZE VALUES
C
        N=NELEV-1
        AMAX=-1.E10
        BMAX=-1.E10
        DTC=DRDA*FSM2FT
C
C  DO FOR ALL PROFILE PTS
C
        IF (N.LT.2) N=2
        DO 40 I=2,N
           DI=DINC*(I-1)
           DC=((DI*DI)/RDA)*FSM2FT
           HC=H(I)-DC
           A=(HC-HTS)/(DI*FSM2FT)
           IF (A.LT.AMAX) GOTO 20
           AMAX=A
           LT=I
 20        B=(HC-HRS+DTC)/((DST-DI)*FSM2FT)
           IF (B.LT.BMAX) GOTO 40
           BMAX=B
           LR=I
 40     CONTINUE
C
C  RETURN HEIGHTS AND DISTANCES
C
        IF (LR.GE.LT) GOTO 60
        LT=NELEV
        LR=1
 60     HLT=H(LT)
        DLT=DINC*(LT-1)
        HLR=H(LR)
        DLR=DST-DINC*(LR-1)
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE EFFHGT(H)
C
C  EFFHGT COMPUTES THE EFFECTIVE TX AND RX ANTENNA HEIGHTS.
C
C  THE EFFECTIVE ANTENNA HEIGHT IS DEFINED AS THE MAXIMUM OF :
C  THE STRUCTURAL HEIGHT ABOVE TERRAIN, AND THE CORRECTED
C  HEIGHT ABOVE THE LOCAL AVERAGE TERRAIN.
C
C          INPUT
C             LT10    -  TEN-MILE POINT FROM TX
C             LR10    -  TEN-MILE POINT FROM RX
C             LT      -  TX HORIZON POINT
C             LR      -  RX HORIZON POINT
C             H       -  PROFILE ELEVATIONS
C             HST     -  STRUCTURAL TX ANTENNA HEIGHT
C             HSR     -  STRUCTURAL RX ANTENNA HEIGHT
C             HTS     -  TX ANTENNA ELEV
C             HRS     -  RX ANTENNA ELEV
C
C          OUTPUT
C             HTE     -  TX EFFECTIVE ANTENNA HEIGHT
C             HRE     -  RX EFFECTIVE ANTENNA HEIGHT
C
        REAL H(*)
        COMMON /PROFIL/NELEV,DINC
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /GEOM2/LT10,LR10,RDA,DRDA
C
C  BEGIN
C
        N1=NELEV-1
        HTAV=0.
        L=MIN(LT-1,LT10)
        IF (L.LT.2) GOTO 40
        SUMHT=0.
        DO 20 I=2,L
           SUMHT=SUMHT+H(I)
 20     CONTINUE
        AV1=SUMHT/(L-1)
        HTAV=HTS-AV1
 40     HTE=MAX(HTAV,HST)
        HRAV=0.
        M=MAX(LR+1,LR10)
        IF (M.GE.NELEV) GOTO 80
        SUMHR=0.
        DO 60 I=N1,M,-1
           SUMHR=SUMHR+H(I)
 60     CONTINUE
        AV2=SUMHR/(NELEV-M)
        HRAV=HRS-AV2
 80     HRE=MAX(HRAV,HSR)
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE TRAYTR(H1,H2)
C
C  RAYTR COMPUTES A CORRECTION TO THE ANTENNA HEIGHT BASED ON THE
C  NATIONAL BUREAU OF STANDARDS EXPONENTIAL REFERENCE ATMOSPHERE
C  MODEL.
C  SOURCE : BEAN,B.R. AND THAYER,G.D.,"CRDL EXPONENTIAL REFERENCE
C  ATMOSPHERE",NBS MONOGRAPH 4, OCT 20,1959
C
C          INPUT
C            H1     -  EFFECTIVE ANTENNA HEIGHT
C            H2     -  ANTENNA ELEV
C            XN     -  REFRACTIVITY
C            RDA    -  EARTH CURVATURE CORRECTION FACTOR
C
C          OUTPUT
C            H2     -  CORRECTED ANTENNA ELEV
C
        DIMENSION TABVAL(24,8),TABH(24),TABN(8),TAB1(96),TAB2(96)
        COMMON /GEOM2/LT10,LR10,RDA,DRDA
        COMMON /CONST/FSM2FT,FM2FT,FKM2SM,FNM2SM,FR2D,PI
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
        EQUIVALENCE (TABVAL,TAB1),(TABVAL(1,5),TAB2)
C
C  DATA TABLES
C
        DATA TAB1/.0011714,.0023433,.0058628,.011742,.023534,.035993,
     & .05931,.083461,.12012,.18546,.24611,.38395,.65514,.94881,
     & 1.4158,3.1409,5.1265,9.0953,13.379,17.806,22.342,49.83,81.657,
     & 115.1,5.877E-4,.0011758,.0029456,.0059175,.011915,.018337,
     & .030554,.043465,.063533,.10067,.13646,.22209,.4041,.61845,
     & .97086,2.4024,4.1519,7.7745,11.77,15.948,20.26,46.726,77.687,
     & 110.38,-5.166E-6,-7.416E-6,1.225E-6,5.224E-5,2.4351E-4,
     & 6.3174E-4,.0017845,.0035319,.007191,.016631,.028203,.063506,
     & .16119,.2951,.55035,1.724,3.2708,6.599,10.35,14.316,18.439,
     & 44.037,74.263,106.33,-5.1424E-4,-.0010273,-.0025545,-.0050557,
     & -.0099458,-.014835,-.023345,-.031325,-.042035,-.056544,-.065784,
     & -.073319,-.046267,.024347,.19795,1.1676,2.5569,5.6578,9.2212,
     & 13.025,17.003,47.697,71.613,103.21/
        DATA TAB2/-.0014311,-.0028607,-.0071318,-.014182,-.028106,
     & -.042354,-.067942,-.093047,-.12862, -.18471,-.22981,
     & -.31035,-.40153,-.43498,-.39341,.25305,1.3955,4.1413,
     & 7.411,10.962,14.712,38.617,67.417,98.271,-.0028411,-.0056795,
     & -.014164,-.028194,-.055947,-.084471,-.13599,-.18696,-.26,
     & -.37781,-.47552,-.66141,-.91885,-1.0951,-1.2309,-1.0083,
     & -.18594,2.0999,4.9897,8.2131,11.67,34.239,61.913,91.804,
     & -.0046404,-.0092765,-.023135,-.046048,-.091356,-.13792,
     & -.22203,-.30525,-.4246,-.61764,-.77853,-1.0885,-1.5362,-1.8716,
     & -2.201,-2.4324,-1.9503,-.15326,2.3326,5.2075,8.3525,29.501,
     & 55.978,84.845,-.010789,-.021565,-.05374,-.10679,-.21126,
     & -.31797,-.50915,-.69655,-.96252,-1.3865,-1.7355,-2.4012,
     & -3.3663,-4.1164,-4.9361,-6.2916,-6.6485,-6.0574,-4.5685,
     & -2.5541,-.17916,17.499,40.974,67.306/
        DATA TABH/.01,.02,.05,.1,.2,.305,.5,.7,1.0,1.524,2.0,3.048,
     & 5.0,7.0,10.0,20.0,30.48,50.0,70.0,90.0,110.0,225.0,350.0,475.0/
        DATA TABN/200.0,252.9,289.0,313.0,344.5,377.2,404.8,450.0/
        DATA MAXNH/24/
        DATA MAXNN/8/
C
C  BEGIN
C
        FKM2FT=3280.8334
        IF (H1.LT.FKM2FT) GOTO 100
        HGT=H1/FKM2FT
        DELH=0.
        HGT=MIN(HGT,475.)
C
C  COMPUTE CORRECTION FROM TABLES
C
        DO 20 IH=1,MAXNH
           IF (TABH(IH).GT.HGT) GOTO 40
 20     CONTINUE
 40     DIFH1=HGT-TABH(IH-1)
        DIFH2=TABH(IH)-TABH(IH-1)
        RATIOH=DIFH1/DIFH2
        DO 60 IN=1,MAXNN
           IF (TABN(IN).GT.XN) GOTO 80
 60     CONTINUE
 80     DIFN1=XN-TABN(IN-1)
        DIFN2=TABN(IN)-TABN(IN-1)
        RATION=DIFN1/DIFN2
        HDIF1=TABVAL(IH,IN-1)-TABVAL(IH-1,IN-1)
        HDIF2=TABVAL(IH,IN)-TABVAL(IH-1,IN)
        HTAB1=HDIF1*RATIOH
        HTAB2=HDIF2*RATIOH
        VALBH1=TABVAL(IH-1,IN-1)+HTAB1
        VALBH2=TABVAL(IH-1,IN)+HTAB2
        DIF=VALBH2-VALBH1
        VALBN=DIF*RATION
        DELH=VALBH1+VALBN
C
C  CORRECT FOR EARTH CURVATURE
C
        DELFT=DELH*FKM2FT
        HGTA=H1-DELFT
        DLA=SQRT(HGTA+HGTA)
        TXD=H1-(DLA*DLA*FSM2FT)/RDA
        H2=H2-TXD
C
C  ALL DONE, RETURN
C
 100    RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE ANGDST(H)
C
C  ANGDST COMPUTES THE VERTICAL ANGLES ALPH00,BETA00,AND THE
C  ANGULAR DISTANCE THET00.
C
C          INPUT
C              HTS     -  TX ANTENNA ELEV
C              HRS     -  RX ANTENNA ELEV
C              DLT     -  DIST TO TX RADIO HORIZON
C              DLR     -  DIST TO RX RADIO HORIZON
C              DST     -  PATH DISTANCE
C              H       -  PROFILE ELEVS
C              RDA     -  CORRECTION FACTOR
C              DRDA    -  CORRECTION FACTOR : DST**2/RDA
C              FSM2FT  -  CONVERSION FACTOR
C
C           OUTPUT
C              ALPH00  -  ANGLE BETWEEN ANT-TO-ANT RAY AND TX HORIZ
C              BETA00  -  ANGLE BETWEEN ANT-TO-ANT RAY AND RX HORIZ
C              THET00  -  PATH ANGULAR DISTANCE = ALPH00+BETA00
C             THET    -  TX RADIO HORIZON ANGLE
C             THER    -  RX RADIO HORIZON ANGLE
C
C
        REAL H(*)
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /GEOM2/LT10,LR10,RDA,DRDA
        COMMON /PROFIL/NELEV,DINC
        COMMON /CONST/FSM2FT,FM2FT,FKM2SM,FNM2SM,FR2D,PI
        COMMON /AKK/AKK(6),T1,T2,THET,THER
C
C  BEGIN
C
        HLT=H(LT)
        HLR=H(LR)
        HTRS=(HRS-HTS)/FSM2FT
        TT2=((HLT-HTS)/(DLT*FSM2FT))-(DLT/RDA)
        THET=ATAN(TT2)
        TT3=(DRDA-HTRS)/DST
        ALPH00=THET+ATAN(TT3)
        TT2=((HLR-HRS)/(DLR*FSM2FT))-(DLR/RDA)
        THER=ATAN(TT2)
        TT3=(HTRS+DRDA)/DST
        BETA00=THER+ATAN(TT3)
        THET00=ALPH00+BETA00
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE PMS
C
C  PMS CALCULATES AN ARRAY OF GEOMETRICAL CONSTANTS
C  LATER USED IN THE PROPAGATION MODELS, AND MAKES
C  THE INITIAL MODEL SELECTION BASED ON DATA COMPUTED
C  BY SUBROUTINE EXTRA.
C
C        INPUT
C           FREQ   -  FREQUENCY
C           DLT    -  DIST TO TX HORIZON
C           DLR    -  DIST TO RX HORIZON
C           DST    -  PATH DISTANCE
C           HTE    -  TX EFFECTIVE ANT HGT
C           HRE    -  RX EFFECTIVE ANT HGT
C           ALPH00 -  TX COMPONENT ANGLE
C           BETA00 -  RX COMPONENT ANGLE
C           THET00 -  PATH ANGULAR DIST
C           RE     -  EARTH RADIUS
C           AK     -  EARTH CURVATURE CORRECTION
C           FSM2FT -  CONVERSION FACTOR
C
C       OUTPUT
C           AKK    -  ARRAY OF GEOMETRICAL PARAMETERS
C           T1     -  GEOMETRICAL PARAMETER
C           T2     -  GEOMETRICAL PARAMETER
C           MODEL  -  INITIAL PROPAGATION MODEL SELECTION
C          T1     -  TX PATH GEOMETRY ANGLE
C          T2     -  RX PATH GEOMETRY ANGLE
C
        COMMON /LOSS/MODEL,PATLOS,PLFS,PLKED,PLRED,PLONG
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
        COMMON /CONST/FSM2FT,FM2FT,FKM2SM,FNM2SM,FR2D,PI
        COMMON /AKK/AKK(6),T1,T2,THET,THER
        DATA THIRD/.333333333/
C
C  BEGIN
C
        T1=((DST*BETA00)/THET00)-DLT
        T2=((DST*ALPH00)/THET00)-DLR
        TAT=DST-DLT-DLR
        IF (TAT.LT.0.0001) GOTO 10
        T3=THET00/TAT
        AKK(1)=(FSM2FT*DLT*DLT)/(2.0*HTE*RE)
        AKK(2)=(FSM2FT*DLR*DLR)/(2.0*HRE*RE)
        AKK(5)=T2*T3
        AKK(3)=T1/(AKK(5)*RE)
        AKK(6)=T1*T3
        AKK(4)=T2/(AKK(6)*RE)
 10     IF (LT.EQ.LR) GOTO 20
        IF (LR.EQ.1) GOTO 30
        THMIN=.03/((.75*AK*FREQ)**THIRD)
        IF (THET00.GE.THMIN) GOTO 40
        IF (THET00.GT.0.) GOTO 50
        GOTO 30
C
C  SINGLE KNIFE-EDGE PATH, BEYOND L-O-S
C
 20     IF (THET00.LE.0.) GOTO 30
        MODEL=2
        GOTO 70
C
C  LINE-OF-SIGHT PATH
C
 30     MODEL=7
        GOTO 70
C
C  BEYOND L-O-S, > 2 SM B/T HORIZONS
C
 40     IF (TAT.LT.2.0) GOTO 60
        MODEL=6
        GOTO 70
C
C  EFFECTIVE KNIFE-EDGE
C
 50     MODEL=1
        GOTO 70
C
C  BEYOND L-O-S, < 2 SM B/T HORIZONS
C
 60     MODEL=4
 70     RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE FSLOSS
C
C  FSLOSS COMPUTES THE FREE-SPACE PATH LOSS
C
C          INPUT
C            RE     -  EARTH RADIUS
C            AK     -  EFFECTIVE EARTH RADIUS FACTOR
C            HTE    -  TX EFFECTIVE ANTENNA HEIGHT
C            HRE    -  RX EFFECTIVE ANTENNA HEIGHT
C            DST    -  PATH DISTANCE
C            RDA    -  2*AK*RE
C            FSM2FT -  CONVERSION FACTOR
C
C          OUTPUT
C            PLFS   -  FREE-SPACE PATH LOSS
C
        DOUBLE PRECISION ERHO1,ERHO2,ETHETA
        COMMON /LOSS/MODEL,PATLOS,PLFS,PLKED,PLRED,PLONG
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /GEOM2/LT10,LR10,RDA,DRDA
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
        COMMON /CONST/FSM2FT,FM2FT,FKM2SM,FNM2SM,FR2D,PI
C
C  EFFECTIVE EARTH RADIUS
C
        AMI=RE*AK
        HTEM=HTE/FSM2FT
        HREM=HRE/FSM2FT
C
C  TX AND RX HEIGHTS RELATIVE TO EARTH CENTER, PATH CENTRAL ANGLE
C
        ERHO1=HTEM+AMI
        ERHO2=HREM+AMI
        ETHETA=DST/AMI
        DRLOS=SQRT(HTEM*(HTEM+RDA))+SQRT(HREM*(HREM+RDA))
        IF (DST.GT.DRLOS) GOTO 10
        SLNTR=DSQRT((ERHO1**2)+(ERHO2**2)-(2.*ERHO1*ERHO2*DCOS(ETHETA)))
        GOTO 20
 10     SLNTR=(AMI+(.5*(HTEM+HREM)))*ETHETA
        SLNTR=AMAX1(DRLOS,SLNTR)
C
C  COMPUTE LOSS
C
 20     PLFS=36.58+(20.*ALOG10(FREQ*SLNTR))
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE KED(H)
C
C  KED COMPUTES PROPAGATION LOSS FOR SINGLE KNIFE-EDGE PATHS.
C
C            INPUT
C              NELEV  - NO OF POINTS IN PROFILE
C              LT     - TX RADIO HORIZON PT
C              LR     - RX RADIO HORIZON PT
C              DST    - PATH DISTANCE (SM)
C              FREQ   - FREQUENCY (MHZ)
C              ALPH00 - ANGLE BETWEEN RAY CONNECTING ANTS & TX HORIZ (RAD)
C              BETA00 - ANGLE BETWEEN RAY CONNECTING ANTS & RX HORIZ (RAD)
C              PLFS   - FREE-SPACE LOSS (DB)
C
C            OUTPUT
C              PLKED  - KNIFE-EDGE DIFFRACTION LOSS (DB)
C
C            MAJOR LOCAL VARIABLES
C              W      - DIFFRACTION LOSS (DB)
C              TXHTG  - TX HEIGHT GAIN (DB)
C              RXHTG  - RX HEIGHT GAIN (DB)
C
C            SUBROUTINE CALLED
C              HTGAIN - COMPUTES HEIGHT GAIN
C
        REAL H(*)
        COMMON /LOSS/MODEL,PATLOS,PLFS,PLKED,PLRED,PLONG
        COMMON /PROFIL/NELEV,DINC
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
C
C  COMPUTE DIFFRACTION LOSS
C
        V=3.28*SQRT(DST*FREQ*TAN(ALPH00)*TAN(BETA00))
        IF (V.LT.2.5) GOTO 10
        W=12.95+(20.*ALOG10(V))
        GOTO 20
 10     W=6.0+(9.4*V)-(1.4*(V**2))
C
C  COMPUTE HEIGHT GAIN FOR TX
C
 20     NST=2
        NEND=LT-1
        IF (LT.EQ.2) NEND=NST
        CALL HTGAIN(H,NST,NEND,TXHTG)
C
C  COMPUTE HEIGHT GAIN FOR RX
C
        NST=LR+1
        NEND=NELEV-1
        IF (NST.GT.NEND) NST=NEND
        CALL HTGAIN(H,NST,NEND,RXHTG)
C
C  COMPUTE KNIFE-EDGE DIFFRACTION LOSS
C
        PLKED=PLFS+W+TXHTG+RXHTG
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE HTGAIN(H,NST,NEND,HTGN)
C
C  HTGAIN EXAMINES THE PROFILE FROM EITHER THE TX OR RX AND
C  COMPUTES THE CORRESPONDING HEIGHT GAIN. IF THE PROFILE
C  DOES NOT INTERSECT THE FIRST FRESNEL ZONE, NO HEIGHT
C  GAIN IS CALCULATED.
C
C           INPUT
C             NST    -  FIRST PROFILE PT EXAMINED
C             NEND   -  LAST PROFILE PT EXAMINED
C             AK     -  CORRECTION FOR EARTH CURVATURE
C             DST    -  PATH DISTANCE
C             H      -  PROFILE ELEVATIONS
C             DINC   -  PROFILE INCREMENT
C             NELEV  -  NO OF PTS IN PROFILE
C             FREQ   -  FREQUENCY
C             DLT    -  DIST TO TX RADIO HORIZON
C             DLR    -  DIST TO RX RADIO HORIZON
C             LT     -  TX HORIZON POINT
C             LR     -  RX HORIZON POINT
C             HTS    -  TX ANTENNA ELEV
C             HRS    -  RX ANTENNA ELEV
C             HTE    -  TX EFFECTIVE ANT HGT
C             HRE    -  RX EFFECTIVE ANT HGT
C
C          OUTPUT
C             HTGN   -  HEIGHT GAIN
C
        REAL H(*)
        DIMENSION C(3,2)
        COMMON /PROFIL/NELEV,DINC
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
        DATA TTHRDS/.666666667/
        DATA C/7.31675,0.42177,0.89142,8.06034,-2.54869,4.12806/
C
C  SET LOGIC BRANCHES AND INITIALIZE VARIABLES
C
        HTGN=0.0
        HLT=H(LT)
        HLR=H(LR)
C
C  ISW=1 -> CALCULATE FROM TX, ISW=2 -> CALCULATE FROM RX
C
        ISW=1
        IF (NST.NE.2) ISW=2
C
C  JSW=1 -> NO PTS BETWEEN NST & NEND, JSW=2 -> GENERAL CASE
C
        JSW=1
        IF (NST.NE.NEND) JSW=2
        IF (ISW.EQ.2) GOTO 5
        DLQ=DLT
        HQE=HTE
        GOTO 10
 5      DLQ=DLR
        HQE=HRE
 10     AKD=TTHRDS/AK
        DLT2K=AKD*(DLT**2)
        DST2K=AKD*(DST**2)
        ICOUNT=0
        NPTS=NEND-NST+1
        IF (NPTS.EQ.0) GOTO 150
C
C  LOOP THROUGH SPECIFIED PROFILE POINTS
C
        DO 120 I=NST,NEND
           XI=(I-1)*DINC
           YI=H(I)-(AKD*(XI**2))
           GOTO (20,60) JSW
 20        IF (ISW.EQ.2) GOTO 40
C
C  INTERPOLATE ONE POINT BETWEEN TX AND TX HORIZON
C
           XI=XI/2.
           YI=H(1)+(YI-H(1))/2.
           GOTO 80
C
C  INTERPOLATE ONE POINT BETWEEN RX AND RX HORIZON
C
 40        XI=XI+(DST-XI)/2.
           YN=H(NELEV)-DST2K
           YI=YN+(YI-YN)/2.
           GOTO 100
 60        GOTO (80,100) ISW
C
C  CALCULATE FOR TX
C
 80        RIX=XI
           RIY=DLT-XI
           RI=2280.*SQRT((RIX*RIY)/(DLT*FREQ))
           TEMP=HLT-HTS-DLT2K
           ALI=HTS+(RIX*TEMP/DLT)
           GOTO 110
C
C  CALCULATE FOR RX
C
 100       RIX=DST-XI
           RIY=DLR-RIX
           RI=2280.*SQRT((RIX*RIY)/(DLR*FREQ))
           TEMP=HRS-DST2K
           ALI=RIX*(HLR-(AKD*(DST-DLR)**2)-TEMP)/DLR+TEMP
C
C  FRESNEL ELLIPSE INTERSECTED
C
 110       IF (YI.GT.(ALI-RI)) ICOUNT=ICOUNT+1
 120    CONTINUE
        IF (ICOUNT.EQ.0) GOTO 150
        RATIOG=ICOUNT/(NPTS/2.)
        IF (RATIOG.GT.1.0) RATIOG=1.0
C
C  COMPUTE HEIGHT GAIN
C
 130    HEBAR=(((HQE**2)*FREQ)/(9.03E5*DLQ))**TTHRDS
        Z=ALOG10(HEBAR)
        K=1
        IF (HEBAR.LT.2.) K=2
        HTGN=C(1,K)+(C(2,K)*Z)+(C(3,K)*Z*Z)
        HTGN=HTGN*RATIOG
C
C  ALL DONE
C
 150    RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE RED
C
C      THIS SUBROUTINE CALCULATES THE ROUGH-EARTH DIFFRACTION
C      LOSS FOR THE *TIREM* MODEL.
C      THE BASIS FOR THE CALCULATIONS IS K.A. NORTON'S PREDICTION METHODS
C      USING BREMMER'S COMPLEX INFINITE SERIES EXPANSION EQUATIONS.
C
C      PROGRAMMING IS BASED ON THE SUBROUTINE OF THE SAME NAME IN *MODLIB* AT
C      ECAC, AND USES THE SAME VARIABLE NAMES AND 'STYLE'. HOWEVER,
C      COMMON BLOCKS HAVE BEEN RE-NAMED AND RE-ARRANGED.
C
C        INPUT
C      VARIABLES
C                        AKK - 'K' FACTORS, EFFECTIVE EARTH'S RADII AT
C                              DIFFERENT PARTS OF THE PROBLEM GEOMETRY
C                        DST - DISTANCE BETWEEN THE TX AND RX (ST. MI.)
C                        EPS - PERMITTIVITY
C                       FREQ - FREQUENCY OF TX (MHZ)
C                       FR2D - RADIANS TO DEGREES CONVERSION FACTOR
C                        HRE - RX EFFECTIVE ANTENNA HEIGHT (FT)
C                        HTE - TX EFFECTIVE ANTENNA HEIGHT (FT)
C                       IPOL - ANTENNA POLARIZATION
C                         PI - THE CONSTANT 'PI'
C                         RE - EARTH'S RADIUS (ST. MI.)
C                      SIGMA - CONDUCTIVITY
C
C       OUTPUT
C      VARIABLES
C                      PLRED - ROUGH-EARTH DIFFRACTION LOSS
C
C      MAJOR LOCAL
C      VARIABLES
C                      ALPH0 - REAL PORTION OF COMPLEX VARIABLE USED IN BREMMER
C                              EXPANSION (OTHER 'ALPH-' VARIABLES ARE
C                              VARIATIONS)
C                          B - MAJOR 'EARTH CONSTANTS' PARAMETER
C                      BETA0 - IMAGINARY PORTION OF COMPLEX VARIABLE USED IN
C                              BREMMER EXPANSION (OTHER 'BETA-' VARIABLES
C                              ARE VARIATIONS)
C                         CO - THE 'C' FUNCTION FOR EACH 'K' FACTOR
C                     DELTAI - HEIGHT GAIN PARAMETER
C                     DFHBRA - HEIGHT GAIN FUNCTION FOR RX AND TX
C                      GAMAI - GAMMA FUNCTION IN BREMMER EXPANSION
C                       GHBR - ANTENNA HEIGHT GAIN FUNCTION
C                         HA - HANKEL FUNCTIONS
C                      HALFC - HALF THE 'C' FUNCTION
C                       TAU0 - THE GENERAL TERM OF THE COMPLEX VARIABLE
C                              FOR THE BREMMER EXPANSION
C                      THETA - ANGULAR PATH DISTANCES BETWEEN RADIO HORIZONS
C                          X - CONDUCTIVITY-WAVELENGTH PARAMETER
C                       YABS - 'Y' FACTOR TO DETERMINE HEIGHT GAIN
C
C
        DOUBLE PRECISION DT1,DT5,DTEMP,DFHBRA
        DIMENSION THETA(2),AK(4),GHBR(2),HALFC(2),ALPH0(2),BETA0(4),
     &       C0(4),TAU0(2)
        COMMON /AKK/AKK(6),ZT1,ZT2,THET,THER
        COMMON /CONST/FSM2FT,FM2FT,FKM2SM,FNM2SM,FR2D,PI
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /LOSS/MODEL,PATLOS,PLFS,PLKED,PLRED,PLONG
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,ZAK,KEPM
        DATA THIRD/.333333/
C
C  CALCULATE PARAMETERS TO BE USED WHICH ARE BASED
C  ON EARTH CONSTANTS, FREQUENCY, ETC.
C
        TFRCUB=FREQ**THIRD
        TFRCB2=TFRCUB**2*3.3346E-5
        CUBPI=(4.*PI**2)**THIRD
        X=18000./FREQ*SIGMA
        BP=ATAN((EPS-1)/X)
        COSBP=COS(BP)
C
C  CALCULATE FACTORS FOR VERTICAL POLARIZATION
C
        IF (IPOL.EQ.0) GOTO 10
        BPP=ATAN(EPS/X)
        COSB=COS(BPP)
        QT=(X*COSBP)/COSB**2
        GOTO 20
C
C  CALCULATE FACTORS FOR HORIZONTAL POLARIZATION
C
 10     BPP=PI/2
        QT=COSBP/X
C
C  CALCULATE SINE-COSINE FACTORS FOR BREMMER SERIES COMPUTATIONS
C
 20     B=2.*BPP-BP
        S1=135./FR2D-.5*B
        S3=105./FR2D-1.5*B
        S5=75./FR2D-3.5*B
        S6=150./FR2D-3.*B
        S7=225./FR2D-3.5*B
        S8=120./FR2D-4.*B
C
        COS1=COS(S1)
        COS3=COS(S3)
        COS4=-COS(2*B)
        COS5=COS(S5)
        COS6=COS(S6)
        COS7=COS(S7)
        COS8=COS(S8)
C
        SIN1=SIN(S1)
        SIN3=SIN(S3)
        SIN4=SIN(2*B)
        SIN5=SIN(S5)
        SIN6=SIN(S6)
        SIN7=SIN(S7)
        SIN8=SIN(S8)
C
        S1=.5*B-195./FR2D
        S3=1.5*B-165./FR2D
        S4=2.*B-240./FR2D
        S5=2.5*B-135./FR2D
        S6=3.*B-30./FR2D
        S7=3.5*B-105./FR2D
C
        COS9=COS(S1)
        COS10=COS(B)
        COS11=COS(S3)
        COS12=COS(S4)
        COS13=COS(S5)
        COS14=COS(S6)
        COS15=COS(S7)
        COS16=COS(4*B)
C
        SIN9=SIN(S1)
        SIN10=-COS(B)
        SIN11=SIN(S3)
        SIN12=SIN(S4)
        SIN13=SIN(S5)
        SIN14=SIN(S6)
        SIN15=SIN(S7)
        SIN16=SIN(4*B)
C
        BDEG=B*FR2D
        AKO=SQRT(QT)*((300./FREQ)/(2.*PI*RE*1609.35))**THIRD
C
C  CALCULATE THE COMPONENTS AND VARIABLES FOR THE BREMMER SERIES.
C  CALCULATIONS ARE MADE FOR EACH EFFECTIVE EARTH'S RADIUS (AKCB).
C
        DO 70 I=1,4
           AKCB=ABS(AKK(I))**THIRD
           AKCB=SIGN(AKCB,AKK(I))
           AK(I)=AKO/AKCB
           AKJ=AK(I)
           C0(I)=ABS(0.9085603*AKCB)
           AK4=AKJ-.4
           AK6=AKJ-.6
           IF (AK6.GE.0.0) GOTO 30
           IF (AK4.GT.0.0) AKJ=0.4
C
C  COMPUTE COEFFECIENTS FOR THE REAL AND IMAGINARY COMPONENTS
C
           T3=1.2371723*AKJ**3
           T4=0.5*AKJ**4
           T5=2.7550715*AKJ**5
           T6=2.8867355*AKJ**6
           T7=6.589639*AKJ**7
           T8=13.316179*AKJ**8
C
C  CALCULATE INTERMEDIATE REAL AND IMAGINARY COMPONENTS OF THE COMPLEX TERM
C
           ALPH4=0.92787925-(AKJ*COS1)-(T3*COS3)+(T4*COS4)-
     &          (T5*COS5)+(T6*COS6)+(T7*COS7)+(T8*COS8)
           BETA4=1.607133978-(AKJ*SIN1)-(T3*SIN3)+(T4*SIN4)-
     &          (T5*SIN5)+(T6*SIN6)+(T7*SIN7)+(T8*SIN8)
           IF (AK4.LE.0.0) GOTO 40
           AKJ=0.6
C
C  COMPUTE COEFFECIENTS FOR REAL AND IMAGINARY COMPONENTS
C
 30        T1=0.61834008/AKJ
           T2=0.2364189/AKJ**2
           T3=0.053338757/AKJ**3
           T4=0.0022553514/AKJ**4
           T5=0.0024693571/AKJ**5
           T6=0.00039940653/AKJ**6
           T7=0.000212015/AKJ**7
           T8=0.00046148795/AKJ**8
C
C  CALCULATE INTERMEDIATE REAL AND IMAGINARY COMPONENTS OF THE COMPLEX TERM
C
           ALPH6=0.404308258-(T1*COS9)-(T2*COS10)+(T3*COS11)-
     &          (T4*COS12)+(T5*COS13)+(T6*COS14)+(T7*COS15)-(T8*COS16)
           BETA6=0.700282434-(T1*SIN9)-(T2*SIN10)+(T3*SIN11)-
     &          (T4*SIN12)+(T5*SIN13)+(T6*SIN14)+(T7*SIN15)-(T8*SIN16)
           IF (AK6.LT.0.0) GOTO 50
           ALPHI=ALPH6
           BETAI=BETA6
           GOTO 60
 40        ALPHI=ALPH4
           BETAI=BETA4
           GOTO 60
 50        ALPHI=(AK4*(ALPH6-ALPH4)/0.2)+ALPH4
           BETAI=(AK4*(BETA6-BETA4)/0.2)+BETA4
 60        BETA0(I)=BETAI
           IF (I.GT.2) GOTO 70
C
C  CALCULATE THE COMPLEX TERM AND ANGULAR DISTANCES
C
           TAU0(I)=SQRT(ALPHI**2+BETAI**2)
           THETA(I)=ATAN(BETAI/ALPHI)
           ALPH0(I)=ALPHI
 70     CONTINUE
C
C  CALCULATE SINE-COSINE FACTORS FOR INTERMEDIATE FORMULATIONS
C
        BSIN=.5*SIN(B)
        S1=45./FR2D-.5*B
        S2=120./FR2D-B
        S3=195./FR2D-1.5*B
        COS17=COS(S1)
        COS18=COS(S2)
        COS19=COS(S3)
        COS20=COS(2*B)
        SIN17=SIN(S1)
        SIN18=SIN(S2)
        SIN19=SIN(S3)
        SIN20=SIN(2*B)
        S1=B/2-225./FR2D
        S2=B-300./FR2D
        COS21=COS(S1)
        COS22=COS(S2)
        SIN21=SIN(S1)
        SIN22=SIN(S2)
        BCOS=.5*COS(B)
C
C  IN THIS SECTION, CALCULATIONS ARE MADE USING THE FACTORS OF THE
C  BREMMER SERIES TO FIND HEIGHT GAIN FUNCTIONS AND HANKEL EQUATIONS
C
        DO 200 I=1,2
           AKI=AK(I)
           AKJ=AKI
           BETAI=BETA0(I)
           TAUI=TAU0(I)
           COI=C0(I)
           HEI=HTE
           IF (I.EQ.2) HEI=HRE
           ALPHI=ALPH0(I)
           AK9=0.9-AKI
           AK11=1.1-AKI
           AK4=AKJ-.4
           AK58=AKJ-.58
           IF (AK58.GE.0.0) GOTO 80
           IF (AK4.GT.0.0) AKJ=.4
           T1=2.001475*AKJ
           T2=0.53926065*AKJ**2
           T3=3.4963134*AKJ**3
           T4=0.23571227*AKJ**4
           XK4=(T1*COS17)+(T2*COS18)-(T3*COS19)+(T4*COS20)
           YK4=(T1*SIN17)+(T2*SIN18)-(T3*SIN19)+(T4*SIN20)
           IF (AK4.LE.0.0) GOTO 90
           AKJ=0.58
 80        T1=0.702892554/AKJ
           T2=0.10349714/AKJ**2
           XK58=1.59207963+(T1*COS21)-(T2*COS22)
           YK58=0.9191876+(T1*SIN21)-(T2*SIN22)
           IF (AK58.LT.0.0) GOTO 100
           XK=XK58
           YK=YK58
           GOTO 110
 90        XK=XK4
           YK=YK4
           GOTO 110
 100       XK=(AK4*(XK58-XK4)/0.18)+XK4
           YK=(AK4*(YK58-YK4)/0.18)+YK4
C     
C  COMPUTE HANKEL EQUATIONS AND RELATED FACTORS
C
 110       HAI=SQRT(XK**2+YK**2)
           DELTAI=SQRT(BETAI/(2.*TAUI))/HAI
           HBRI=(TFRCB2*HEI*BETAI**2)/COI
           THETAI=THETA(I)
           HBAR=HBRI
           BARH=HBRI
           IF (AKI.LT.0.8) GOTO 120
           IRG=1
           IF (HBRI.LE.0.1) GOTO 150
           IRG=2
           IF (HBRI.GE.0.3) GOTO 160
           IRG=7
           HBAR=0.3
           BARH=0.1
           GOTO 140
 120       IF (AKI.GT.0.4) GOTO 130
           IRG=3
           IF (HBRI.LE.0.5) GOTO 150
           IRG=4
           IF (HBRI.GE.0.7) GOTO 160
           IRG=9
           HBAR=0.7
           BARH=0.5
           GOTO 140
 130       IRG=5
           IF (HBRI.LE.AK9) GOTO 150
           IRG=6
           IF (HBRI.GE.AK11) GOTO 160
           IRG=8
           HBAR=AK11
           BARH=AK9
 140       I1=1
           I2=1
           GOTO 170
 150       I2=2
           GOTO 180
 160       I1=2
C
C  CALCULATE 'Y' FUNCTION FOR HEIGHT GAIN
C
 170       HBRL=BARH
           HBRH=HBAR
           QK=(CUBPI*HBAR)/BETAI**2
           TAU2=-2.*TAUI
           YR=(TAU2*COS(THETAI))+(2.*QK)
           YI=TAU2*SIN(THETAI)
           YABS=SQRT(YR**2+YI**2)
           GAMMA=-ABS(ATAN2(YI,YR))
           GAMMA5=1.5*GAMMA
           SIN23=SIN(GAMMA5)
           COS23=COS(GAMMA5)
           T4=YABS**3
           T3=SQRT(T4)
           DT1=(T3/3.)*SIN23
           DTEMP=SQRT(6./(PI*BETAI))/YABS**.25
           DT5=DTEMP*DEXP(-DT1)
           T1=0.20833330/T3
           T2=0.33420138/T4
           T3=1.02581257/T3**3
           T4=4.66958443/T4**2
           S2=2.*GAMMA5
           S3=PI/2-3*GAMMA5
           S4=2.*S2
           COSS4=COS(S4)
C
C  CALCULATE INTERMEDIATE VALUES FOR LATER HEIGHT GAIN COMPUTATIONS
C
           FHBRR=1.-(T1*SIN23)-(T2*COS(S2))+(T3*COS(S3))+(T4*COSS4)
           FHBRI=(T2*SIN(S2))-(T1*COS23)+(T3*SIN(S3))-(T4*COSS4)
           DFHBRA=DT5*SQRT(FHBRR**2+FHBRI**2)
           IF (I1.EQ.2) GOTO 190
           FHBRH=DFHBRA
C
C  CALCULATE PARAMETERS AND SINE-COSINE FACTORS FOR
C  HEIGHT GAIN EQUATIONS
C
 180       QK=(CUBPI*BARH)/BETAI**2
           QI=QK/AKI
           T1=QI**2*AKI**2
           T2=T1*TAUI
           T3=THIRD*T1*QI*AKI
           T4=THIRD*T2*QI
           T5=0.5*T3*QI*AKI*TAUI**2
           T6=T1**2*(5./(72.*TAUI))
           T7=0.5*T3*QI
           T8=T7*2.375/TAUI**3
           S1=.25*PI+.5*B
           S4=S1+THETAI
           S8=S1-3*THETAI
           COS24=COS(S1)
           COS25=COS(THETAI)
           COS26=COS(S4)
           SIN25=SIN(THETAI)
           COS27=COS25**2-SIN25**2
           COS28=COS(S8)
           SIN24=SIN(S1)
           SIN26=SIN(S4)
           SIN27=2.*SIN25*COS25
           SIN28=SIN(S8)
C
C  CALCULATE HEIGHT GAIN EQUATIONS
C
           FHBRR=1.+((T7-QI)*COS24)+((T2+T6)*COS25)-T3-
     &          (T4*COS26)+(T5*COS27)-(T8*COS28)
           FHBRI=((T7-QI)*SIN24)+((T2-T6)*SIN25)-T3-(T4*SIN26)+
     &          (T5*SIN27)-(T8*SIN28)
           DFHBRA=SQRT(FHBRR**2+FHBRI**2)/DELTAI
           IF (I2.EQ.2) GOTO 190
           DFHBRA=((FHBRH-DFHBRA)/.2)*(HBRI-HBRL)+DFHBRA
 190       GHBRI=20.*DLOG10(DFHBRA)-22.66675*SQRT(HBRI)
           X1=ALPHI+(BSIN/AKI**2)
           Y1=BETAI+(BCOS/AKI**2)
           GAMAI=0.030046482*SQRT(BETAI/(X1**2+Y1**2))
           HALFC(I)=8.839-(10.*ALOG10(GAMAI))-(20.*ALOG10(DELTAI))+
     &          (5.*ALOG10(BETAI))
           GHBR(I)=GHBRI
 200    CONTINUE
C
C  CALCULATE THE ROUGH-EARTH DIFFRACTION LOSS
C
        PLRED=HALFC(1)+HALFC(2)+(16.667*ALOG10(FREQ))-GHBR(1)-
     &       GHBR(2)+(10.*ALOG10(C0(3)*C0(4)*DST))+((488.69*TFRCUB)*
     &       ((BETA0(3)*C0(3)*AKK(5))+(BETA0(4)*C0(4)*AKK(6))))
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE ALONG(H)
C
C             ANITA LONGLEY (*ALONG*) MODEL
C
C      THIS MODEL IS DENOTED BY THE NUMBER 7
C      IT IS NOT VALID FOR FREQUENCIES < 40 MHZ.
C
C  USE LEAST SQUARES.
C  FIT THE TERRAIN TO A SMOOTH CURVE AND CALCULATE THE INTERDECILE RANGE OF
C  THE TERRAIN WITH RESPECT TO THIS CURVE.
C
C     INPUT:
C            DRDA-CORRECTION FACTOR (DST**2/(2*EFFECTIVE EARTH'S RADIUS))
C            RDA-2*EFFECTIVE EARTH'S RADIUS
C             DST- DISTANCE
C             FREQ - FREQUENCY
C             PLFS - FREESPACE LOSS
C             HRE - EFFECTIVE RCVR HEIGHT
C             HRS - RCVR STRUCT HT + GROUND ELEV
C             HSR - RCVR STRUCT HT
C             HST - XMTR STRUCT HT
C             HTE - EFFECTIVE XMTR HT
C             HTS - XMTR STRUCT HT + GROUND ELEV
C             H(J) - PROFILE GROUND HT
C            IPOL- POLARIZATION INDICATOR
C             NELEV - NUMBER OF PROFILE ELEV
C            FSM2FT - CONVERSION FACTOR SM TO FT
C            FM2FT - CONVERSION FACTOR SM TO FT
C
C     OUTPUT:
C           PLONG-PROPAGATION LOSS RETURNED
C     MAJOR VARIABLES:
C              DELTAH-TERRAIN PARAM. CHARACTERIZING MEDIAN PROPAGATION
C                     CONDITIONS FOR THE PROFILE (METERS).
C               HG(1) - XMTR STRUCT HT (METERS)
C               HG(2) - RCVR STRUCT HT (METERS)
C              ASUBC,ASUBC1,AND ASUBC2-ATTENUATION BELOW FREESPACE
C              D ARRAY-THE ARRAY THAT STORES THE DEL VALUES THAT ARE LESS THAN
C                      AT LEAST 90% OF THE CALCULATED DELS IN THE D ARRAY
C                      POSITIONS OF D(1) THROUGH D(NT) AND THE VALUES THAT ARE
C                      GREATER THAN AT LEAST 90% OF THE DELS IN D(N2) THROUGH
C                      D(ND).
C              DEL-TERRAIN HEIGHTS ABOVE AND BELOW THE SMOOTH CURVE
C              DLSA-SUM OF SMOOTH EARTH HORIZON DISTANCES
C              R-INTERDECILE RANGE OF TERRAIN HTS ABOVE AND BELOW THE SMOOTH
C                CURVE.
C              RATIO-RATIO OF TERRAIN CLEARANCE TO FRESNEL DIMENSION
C              SLNTR-SLANT RANGE
C              XJ , XI-PROFILE DISTANCES
C
C
C     SUBROUTINES AND FUNCTIONS:
C             QLRA - INITIALIZATION FOR ACR AND VALUES TO BE USED IN TESTING
C                    WHETHER OR NOT TO CALL ACR
C             IQRNG - SORTS ELEMENTS IN THE D ARRAY INTO ASCENDING ORDER
C            PREACR-SETS UP INITIAL VALUES FOR ACR
C            ACR-ATTENUATION BELOW FREE SPACE
C
C  THE MAXD PARAMETER MUST BE 1/5TH OF THE LONGEST POSSIBLE TERRAIN PROFILE,
C  PLUS A BIT TO BE SAFE.  THIS MUST BE KEPT IN SYNC WITH THE NELMAX PARAMETER
C  DEFINED IN MODULE TIREM().
C
        PARAMETER (MAXD=1020)
        REAL H(*)
        COMPLEX ZGND
        COMMON /RAD/AE
        COMMON /DIF/AFO,XHT,AHT,WD1,XD1
        COMMON /PREOUT/DLA,EMD,AED,LLOS,LSCAT,THA,XAE
        COMMON /LOS/WLS
        COMMON /SCAT/AD,RR,ETQ,HOS
        COMMON /GEOM2/LT10,LR10,RDA,DRDA
        COMMON /ORD/ND,D(MAXD)
        COMMON /LOSS/MODEL,PATLOS,PLFS,PLKED,PLRED,PLONG
        COMMON /PROFIL/NELEV,DINC
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
        COMMON /IQLR/DELTAH,HG(5)
        COMMON /ACR1/DLSA,WN,WAE,DL(2),THE(2),ZGND,HZ(2),HE(2)
        COMMON /CONST/FSM2FT,FM2FT,FKM2SM,FNM2SM,FR2D,PI
        COMMON /LSFIX/LSFIX
C
C  BEGIN
C
        RATMIN=.5
        RATMAX=1.5
        RATIO=1000.
        MODEL=7
C
C  'DIF' IS THE HEIGHT OF THE RCVR - THE HT OF THE XMTR, BOTH
C  REFERENCED TO HORIZ
C
        DIF=HRS-(DRDA*FSM2FT)-HTS
        SLNTR=SQRT((DST*DST)+(DIF*DIF/2.78784E07))
        NPRIME=NELEV-1
C
C  FIND MINIMUM CLEARANCE ALONG THE PATH
C
        DO 100 J=2,NPRIME
           XJ=DINC*(J-1)
C
C  COMPUTE FRESNEL DIMENSION IN FEET
C
           SLANTJ=XJ*SLNTR/DST
           HFRESJ=2278.8*SQRT((SLANTJ*(SLNTR-SLANTJ))/(FREQ*SLNTR))
C
C  COMPUTE CLEARANCE AT POINT J: RAY HT - TERRAIN HT, BOTH
C  REFERENCED TO SEA LEV.
C
           HRAYJ=((DIF/DST)*XJ)+HTS+(FSM2FT*XJ*XJ/RDA)
           CLEARJ=HRAYJ-H(J)
           RATIOJ=CLEARJ/HFRESJ
           IF (RATIOJ.LT.RATIO) RATIO=RATIOJ
 100    CONTINUE
C
C IF CLEARANCE IS LARGE ENOUGH USE FREESPACE
C
        IF (RATIO.GE.RATMAX) GOTO 800
C
C  890801 - IF FLAG KEPM INDICATES SO, THE EPM-73 MODEL WILL BE SUBSTITUTED FOR
C  LONGLEY-RICE FOR MODES 5 AND 7 (WHICH BECOME MODES 3 AND 9, RESPECTIVELY)
C
        IF (KEPM.EQ.1) THEN
           CALL EPM73(H,XLOSS)
           ASUBC=XLOSS-PLFS
           MODEL=9
           GOTO 1000
        ENDIF
C
C  START OF LONGLEY-RICE PRELIMINARIES
C
        A=0.0
        B=0.0
        E=0.0
        F=0.0
        IF (NELEV.LT.10) GOTO 150
C
C  THE FOLLOWING CALCULATIONS ARE PERFORMED TO STORE ONLY THE NT SMALLEST
C  DEL VALUES AND THE DEL VALUES THAT EXCEED 90% OF THE CALCULATED DEL VALUES
C  IN THE D ARRAY.
C  NT,N2, AND ND ARE ELEMENT NUMBERS CALCULATED FOR STORAGE IN THE D ARRAY
C  NT: THE D ARRAY ELEMENT D(NT-1) IS THE DEL VALUE THAT IS GREATER THAN
C      ABOUT 10% OF THE OTHER DEL VALUES. D(NT) IS THE NEXT LARGEST DEL.
C
        T=NELEV*0.1
        I=T
        DIF=T-I
        NT=I+1
        T=NELEV-T
        I=T
        DIF1=T-I
        N3=NELEV-I+1
C
C  ND: IS THE TOTAL NUMBER OF D ARRAY ELEMENTS.
C
        ND=NT+N3
C
C  N2: D(N2) IS THE D ARRAY ELEMENT THAT IS GREATER THAN 90% OF THE DEL
C      VALUES.
C
        N2=NT+1
        N1=1
C
C CALCULATE THE SMOOTH CURVE FIT (LEAST SQUARES METHOD)
C
 150    DO 200 I=1,NELEV
           XI=DINC*(I-1)
           A=A+XI
           B=B+XI**2
           E=E+H(I)
           F=F+H(I)*XI
 200    CONTINUE
        C=A*A
        D0=NELEV*B-C
        G=((E*B)-(A*F))/D0
        Q=((NELEV*F)-(A*E))/D0
        DO 300 I=1,NELEV
           Z=G+Q*DINC*(I-1)
C
C  CALCULATE DEL
C
           DEL=H(I)-Z
C
C  IF NELEV IS LESS THAN 10, ALL OF THE DEL VALUES ARE  SAVED AND SORTED
C
           IF (NELEV.LT.10) GOTO 295
           IF (I.GT.ND) GOTO 255
C
C  THE FIRST ND DEL VALUES ARE CALCULATED, STORED, AND SORTED.
C
           D(I)=DEL
           IF (I.NE.ND) GOTO 300
           CALL IQRNG
           GOTO 300
C
C  STORE AND SORT THE LARGEST DEL VALUES
C
 255       DO 270 L=ND,N2,-1
              IF (DEL.LT.D(L)) GOTO 270
              IF (L.EQ.N2) GOTO 265
              NH=N2
              DO 260 K=L,N2,-1
                 D(NH)=D(NH+1)
                 NH=NH+1
 260          CONTINUE
 265          D(L)=DEL
              D(L)=DEL
              GOTO 300
 270       CONTINUE
C
C  SORT AND STORE THE NT SMALLEST DEL VALUES.
C
           DO 290 L=1,NT
              IF (DEL.GT.D(L)) GOTO 290
              IF (L.EQ.NT) GOTO 285
              L1=L+1
              NTA=NT
              DO 280 K1=NT,L1,-1
                 D(NTA)=D(NTA-1)
                 NTA=NTA-1
 280          CONTINUE
 285          D(L)=DEL
              GOTO 300
 290       CONTINUE
           IF (NELEV.GE.10) GOTO 300
 295       D(I)=DEL
 300    CONTINUE
        IF (NELEV.GE.10) GOTO 350
        ND=NELEV
        CALL IQRNG
        XL=D(1)
        XH=D(NELEV)
        GOTO 355
 350    XL=D(NT-1)+(DIF*(D(NT)-D(NT-1)))
        XH=D(N2)+(DIF1*(D(N2+1)-D(N2)))
C
C  CALCULATE THE INTERDECILE RANGE OF DELS
C
 355    R=XH-XL
C
C  CALCULATE DELTAH
C
        DELTAH=(R/(1.0-(0.8*EXP(-0.032186*DST))))/FM2FT
        HME=AMIN1(HTE,HRE)
        ASUBC1=(9.0*(1.0+EXP(-0.01*DELTAH)))-(3.5*ALOG10
     &       ((FREQ*HME/FM2FT)/299.7925))+(0.1126545*DST)
        IF ((FREQ.GT.200.0).OR.(DST.LT.0.6214)) GOTO 400
        IF ((HTE.GT.9842.52).OR.(HRE.GT.9842.52)) GOTO 400
C
C  USE THE 1972 LONGLEY-RICE LINE-OF-SITE FORMULATIONS
C  CONVERT HEIGHTS,DISTANCE TO METRIC UNITS (METERS)
C
        HG(1)=HST/FM2FT
        HG(2)=HSR/FM2FT
        DIST=DST*FSM2FT/FM2FT
        CALL QLRA
        LSFIX=0
        IF (DIST.LT.DLSA) GOTO 500
        LSFIX=1
        GOTO 500
 400    ASUBC=ASUBC1
        GOTO 700
C
C  ACR RETURNS THE LONGLEY-RICE LOSS RELATIVE TO FREE SPACE
C
 500    CALL PREACR
        ASUBC2=ACR(DIST)
        IF (FREQ.LT.150.) GOTO 600
        ASUBC=((200.0-FREQ)*ASUBC2+(FREQ-150.0)*ASUBC1)/50.0
        GOTO 700
 600    ASUBC=ASUBC2
 700    CONTINUE
        IF (ASUBC.LE.0.) GOTO 800
C
C  IF IN EPM-73 AUTOMATIC MODE, COMPARE THE PATH LOSS PREDICTED BY EPM-73 TO
C  THAT PREDICTED BY LONGLEY-RICE, USE WHICHEVER PREDICTS LESS LOSS.
C
        IF (KEPM.EQ.2) THEN
           CALL EPM73(H,XLOSS)
           IF (XLOSS.LT.(ASUBC+PLFS)) THEN
              ASUBC=XLOSS-PLFS
              MODEL=9
           ENDIF
        ENDIF
C
C  IF CLEARANCE IS SMALL ENOUGH, USE ALONG (OR EPM-73) 100%
C
 1000   IF (RATIO.LE.RATMIN) GOTO 900
C
C  IF CLEARANCE IS INTERMEDIATE, BLEND MODES.
C  MARGINAL FRESNEL ZONE INTERSECTION
C  COMBINATION OF LONGLEY-RICE (OR EPM-73) AND FREE SPACE
C
        ASUBC=ASUBC*(RATMAX-RATIO)/(RATMAX-RATMIN)
        IF (MODEL.EQ.7) THEN
           MODEL=5
        ELSE
           MODEL=3
        ENDIF
        GOTO 900
C
C  FREE SPACE PATH
C
 800    ASUBC=0.0
        MODEL=8
C
C  COMBINE PATH LOSS WITH FREE SPACE, RETURN
C
 900    PLONG=PLFS+ASUBC
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE IQRNG
C
C      SORT DELTA ARRAY INTO ASCENDING ORDER
C
C   INPUTS:
C          ND - THE NUMBER OF PROFILE ELEVATIONS TO BE SORTED
C          D - THE UNSORTED DELTA ARRAY
C   OUTPUT:
C         D-THE SORTED DELTA ARRAY
C
C  SEE COMMENTS IN ALONG() DESCRIBING PARAMETER MAXD.
C
        PARAMETER (MAXD=1020)
        COMMON /ORD/ND,D(MAXD)
        DO 30 I=2,ND
           IF (D(I).GE.D(I-1)) GOTO 30
           TEMP=D(I)
           IM=I-1
           DO 10 J=1,IM
              L=I-J
              IF (TEMP.GE.D(L)) GOTO 20
              D(L+1)=D(L)
 10        CONTINUE
           D(1)=TEMP
           GOTO 30
 20        D(L+1)=TEMP
 30     CONTINUE
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE QLRA
C
C  THE LONGELY-RICE AREA PREDICTION MODEL.
C  A COVER ROUTINE FOR FUNCTION ACR.
C  INPUT:
C        AK-CORRECTION FACTOR FOR THE EFFECTIVE EARTH'S RADIUS
C        DELTAH-TERRAIN PARAM. CHARACTERIZING MEDIAN PROPAGATION
C               CONDITIONS FOR THE PROFILE (METERS)
C        FREQ-FREQUENCY (MHZ)
C        HG-XMTR AND RCVR STRUCTURAL HT (METERS)
C        RE-EARTH'S RADIUS (SM)
C        SIGMA-THE GROUND CONDUCTIVITY IN SIEMENS (MHOS) PER METER
C
C  OUTPUT:
C        AE-EFFECTIVE EARTH'S RADIUS IN METERS
C        DL-DISTANCES FROM EACH ANTENNA TO THE CORRESPONDING HORIZON
C        DLSA-THE SUM OF THE SMOOTH EARTH HORIZON DISTANCES
C        HE-EFFECTIVE ANTENNA HTS
C        HZ-THE GROUNDWAVE EFFECTIVE HTS
C        THE-THE HORIZON ELEVATION ANGLES
C        WN-WAVE NUMBER
C        ZGND-THE SURFACE TRANSFER IMPEDANCE
C
        COMPLEX ZGND,WQ
        DIMENSION DLS(2)
        DIMENSION KST(2)
        COMMON /CONST/FSM2FT,FM2FT,FKM2SM,FNM2SM,FR2D,PI
        COMMON /RAD/AE
        COMMON /ACR1/DLSA,WN,WAE,DL(2),THE(2),ZGND,HZ(2),HE(2)
        COMMON /IQLR/DELTAH,HG(5)
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
        DATA THIRD/0.333333333333/
C
C  BEGIN
C
        DELTAH=AMAX1(DELTAH,1.E-06)
        AE=RE*AK*FSM2FT/FM2FT
        HTDIFF=(HTE/FM2FT)-HG(1)
        KST(1)=-1
        IF (HTDIFF.GT.5.0) KST(1)=0
        IF (HTDIFF.GT.10.0) KST(1)=1
        HRDIFF=(HRE/FM2FT)-HG(2)
        KST(2)=-1
        IF (HRDIFF.GT.5.0) KST(2)=0
        IF (HRDIFF.GT.10.0) KST(2)=1
        DO 45 I=1,2
           Q=0.
           IF (KST(I))40,25,30
 25        Q=4.
           GOTO 35
 30        Q=9.
 35        IF (HG(I).LT.5.) Q=Q*SIN(0.31459265*HG(I))
           Q=(1.+Q)*EXP(-2.*HG(I)/DELTAH)
 40        HE(I)=HG(I)+Q
           Q=EXP(-0.07*SQRT(DELTAH/AMAX1(HE(I),5.)))
           DL(I)=SQRT(2.*AE*HE(I))*Q
           THE(I)=(0.65*DELTAH*(1.-Q)-2.*HE(I)*Q)/DL(I)
 45     CONTINUE
        WN=FREQ/47.7
        WAE=(0.5*WN*AE)**THIRD
        WQ=CMPLX(EPS,376.62*SIGMA/WN)
        ZGND=CSQRT(WQ-1.)
        IF (IPOL.NE.0) ZGND=ZGND/WQ
        DO 50 I=1,2
           HZ(I)=CABS(HE(I)+CMPLX(0.,1./WN)/(0.98588/WAE+ZGND))
           DLS(I)=SQRT(2.*AE*HZ(I))
 50     CONTINUE
        DLSA=DLS(1)+DLS(2)
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE PREACR
C
C  CALCULATES INITIAL VALUES FOR ACR
C
C  INPUT:
C         WAE, A NORMALIZED EARTH'S RADIUS
C         AE, THE EFFECTIVE EARTH'S RADIUS
C         DL, THE HORIZON DISTANCE
C         DLSA, THE SUM OF THE SMOOTH EARTH HORIZON DISTANCES
C         THE, THE HORIZON ELEVATION ANGLES
C  OUTPUT:
C         EMD, SLOPE OF THE CURVE OF DIFFRACTION ATTENUATION VERSUS DISTANCE
C         AED, ESTIMATED DIFFRACTION ATTENUATION BELOW FREE SPACE
C         DLA, THE TOTAL HORIZON DISTANCE
C        LLOS AND LSCAT, INITIAL VALUES FOR ACR INPUT
C        THA-TOTAL ELEVATION ANGLE
C        XAE-RATIO OF THE EFFECTIVE EARTH'S RADIUS TO A NORMALIZED
C            EARTH'S RADIUS
C  SUBROUTINES:
C               PREDIF, SETS UP THE INITIAL CONSTANTS FOR ADIFF
C               ADIFF, FUNCTION THAT COMPUTES THE DIFFRACTION ATTENUATION
C
        COMPLEX ZGND
        COMMON /ACR1/DLSA,WN,WAE,DL(2),THE(2),ZGND,HZ(2),HE(2)
        COMMON /PREOUT/DLA,EMD,AED,LLOS,LSCAT,THA,XAE
        COMMON /RAD/AE
C
C  BEGIN
C
        LLOS=0
        LSCAT=0
        DLA=DL(1)+DL(2)
        THA=AMAX1(THE(1)+THE(2),-DLA/AE)
C
C  COEFFICIENTS FOR DIFFRCTION RANGE
C
        XAE=AE/WAE
        D3=AMAX1(1.0943*XAE+DLA,DLSA)
        D4=2.1886*XAE+D3
        CALL PREDIF
        A3=ADIFF(D3)
        A4=ADIFF(D4)
        EMD=(A4-A3)/(D4-D3)
        AED=A3-EMD*D3
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE PREDIF
C
C  THIS SUBROUTINE CALCULATES INITIAL CONSTANTS FOR ADIFF
C
C  INPUT:
C        DELTAH-TERRAIN PARAM. CHARACTERIZING MEDIAN PROPAGATION
C               CONDITIONS FOR THE PROFILE (METERS)
C        DL-THE HORIZON DISTANCES
C        DLA-THE SUM OF THE HORIZON DISTANCES
C        DLSA-THE SUM OF THE SMOOTH EARTH HORIZON DISTANCES
C        HE-EFFECTIVE ANTENNA HTS
C        HG-THE STRUCTURAL ANTENNA HTS IN METERS
C        HZ-THE GROUNDWAVE EFFECTIVE HTS
C        AE-EFFECTIVE EARTH'S RADIUS
C        THA-THE TOTAL ELEVATION ANGLE
C        WN-WAVE NUMBER
C        ZGND-THE (COMPLEX) SURFACE TRANSFER IMPEDANCE.
C
C  OUTPUT:
C        AFO-CLUTTER FACTOR
C        AHT,XD1,XHT,WD1-INITIAL CONSTANTS FOR ADIFF
C
C  FUNCTIONS:
C           AWEIG-CALCULATES THE FIRST EIGENVALUE OF THE AIRY FUNCTION
C           FHZ-CALCULATES THE HT GAIN OVER A SMOOTH SPHERICAL EARTH
C               TO BE USED IN THE THREE RADII METHOD.
C
        COMPLEX ZGND
        COMMON /ACR1/DLSA,WN,WAE,DL(2),THE(2),ZGND,HZ(2),HE(2)
        COMMON /IQLR/DELTAH,HG(5)
        COMMON /PREOUT/DLA,EMD,AED,LLOS,LSCAT,THA,XAE
        COMMON /DIF/AFO,XHT,AHT,WD1,XD1
        COMMON /RAD/AE
        DATA THIRD/0.333333333333/
C
C  BEGIN
C
        WD1=SQRT((HE(1)*HE(2))/(HG(1)*HG(2)))
        XD1=THA*AE+DLA
        Q=(1.-0.8*EXP(-DLSA/50.E03))*DELTAH
        X=0.78*Q*EXP(-(Q/16.)**0.25)
        AFO=AMIN1(15.,2.171*ALOG(1.+HG(1)*HG(2)*WN*X*4.77E-04))
        AHT=2.
        XHT=0.
        DO 10 I=1,2
           A=0.5*DL(I)**2/HE(I)
           WA=(A*WN*0.5)**THIRD
           XHT=AWEIG(CMPLX(0.,WA)*ZGND)*WA*DL(I)/A+XHT
           AHT=FHZ(HZ(I)*WN/WA)+AHT
 10     CONTINUE
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        FUNCTION ADIFF(D)
C
C  THE DIFFRACTION ATTENUATION AT DISTANCE D
C  A CONVEX COMBINATION OF SMOOTH EARTH DIFFRACTION AND DOUBLE
C  KNIFE-EDGE DIFFRACTION
C
C  INITIAL CONSTANTS ARE SET IN PREDIF
C  INPUT:
C        AFO,AHT,XHT,XD1,WD1-INITIAL CONSTANTS
C        D-DISTANCE
C        DELTAH-TERRAIN PARAM. CHARACTERIZING MEDIAN PROPAGATION
C               CONDITIONS FOR THE PROFILE (METERS)
C        DL-THE HORIZON DISTANCES
C        DLA-THE SUM OF THE HORIZON DISTANCES
C        AE-EFFECTIVE EARTH'S RADIUS
C        THA-THE TOTAL ELEVATION ANGLE
C        WN-WAVE NUMBER
C        ZGND-SURFACE TRANSFER IMPEDANCE
C
C  OUTPUT:
C        ADIFF-THE DIFFRACTION ATTENUATION AT DISTANCE D
C  FUNCTIONS:
C           AKNFE-KNIFE-EDGE DIFFRACTION. CALCULATES THE FRESNEL INTEGRAL.
C           AWEIG-CALCULATES THE FIRST EIGENVALUE OF THE AIRY FUNCTION
C
C
        COMPLEX ZGND
        COMMON /ACR1/DLSA,WN,WAE,DL(2),THE(2),ZGND,HZ(2),HE(2)
        COMMON /IQLR/DELTAH,HG(5)
        COMMON /RAD/AE
        COMMON /PREOUT/DLA,EMD,AED,LLOS,LSCAT,THA,XAE
        COMMON /DIF/AFO,XHT,AHT,WD1,XD1
        DATA THIRD/0.333333333333/
C
C  BEGIN
C
        TH=THA+D/AE
        DS=D-DLA
        Q=0.0795775*WN*DS*TH**2
        ADIFF=AKNFE(Q*DL(1)/(DS+DL(1)))+AKNFE(Q*DL(2)/(DS+DL(2)))
        A=DS/TH
        WA=(A*WN*0.5)**THIRD
        X=AWEIG(CMPLX(0.,WA)*ZGND)*WA*TH+XHT
        AR=(2.*X-ALOG(X))*4.343-AHT
        Q=(WD1+XD1/D)*AMIN1(((1.-0.8*EXP(-D/50.E03))*DELTAH*WN),6283.2)
        WD=25.1/(25.1+SQRT(Q))
        ADIFF=(AR-ADIFF)*WD+ADIFF+AFO
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        FUNCTION FHZ(X)
C
C THE HEIGHT GAIN OVER A SMOOTH SPHERICAL EARTH
C  TO BE USED IN THE *THREE RADII* METHOD
C
        FHZ=8.686*ALOG(X)+2.94
        IF (X.LE.0.4277) RETURN
        S=SQRT(X/0.4277)
        W=S*EXP(1.-S)
        FHZ=(1.25*W-0.25)*FHZ+(1.-W)*(11.5022*S-4.24)
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        FUNCTION AWEIG(S)
C
C  THE FIRST EIGENVALUE OF THE AIRY FUNCTION
C
        COMPLEX S
        COMPLEX C1,C2,C3
        DATA C1/(1.16906,2.02487)/,C2/(0.77221,0.)/
        DATA C3/(-0.37898,0.65642)/
        AWEIG=AIMAG((C1*S-C2)/(S+C3))
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        FUNCTION AKNFE(V2)
C
C  KNIFE-EDGE DIFFRACTION
C  THE FRESNEL INTEGRAL AS A FUNCTION OF V**2
C
        IF (V2.GE.5.76) GOTO 10
 5      AKNFE=6.02+9.11*SQRT(V2)-1.27*V2
        GOTO 15
 10     AKNFE=12.953+4.343*ALOG(V2)
 15     RETURN
        END
C
C------------------------------------------------------------------------------
C
        FUNCTION ACR(D)
C
C           THE CALCULATED REFERENCE VALUE OF RADIO ATTENUATION
C           A VERSION (SEPT 1972) OF THE LONGLEY-RICE (1968) AREA
C           PREDICTION MODEL.
C
C           VALID ONLY FOR ...
C             FREQUENCIES BELOW 15 GHZ
C              (NO SKYWAVE IS INCLUDED. FOR FREQUENCIES BELOW 40 MHZ
C              DISTANCES MUST BE SHORT.)
C             ANTENNA HEIGHTS BETWEEN 0.5 M AND 3000 M
C             ELEVATION ANGLES LESS THAN 200 MR
C  INPUT:
C        AED-ESTIMATED DIFFRACTION ATTENUATION BELOW FREESPACE
C        D-DISTANCE (METERS)
C        DELTAH-TERRAIN PARAM. CHARACTERIZING MEDIAN PROPAGATION
C               CONDITIONS FOR THE PROFILE (METERS)
C        DL-THE HORIZON DISTANCES
C        DLA-THE SUM OF THE HORIZON DISTANCES
C        DLSA-THE SUM OF THE SMOOTH EARTH HORIZON DISTANCES
C        EMD-SLOPE OF THE CURVE OF DIFFRACTION ATTENUATION VERSUS DISTANCE
C        FREQ-FREQUENCY
C        H(Z)-THE GROUNDWAVE EFFECTIVE HTS
C        LLOS AND LSCAT-INDICATES WHETHER ALOS OR ASCAT HAVE BEEN USED
C        WN-WAVE NUMBER
C        XAE-THE RATIO OF THE EFFECTIVE EARTH'S RADIUS TO A NORMALIZED
C            EARTH'S RADIUS
C        XN-THE SURFACE REFRACTIVITY
C  OUTPUT:
C        ACR(D)-ATTENUATION BELOW FREE SPACE
C  OTHER VARIABLES:
C        WLS-INITIALIZED FOR ALOS
C        AD,ETQ,HOS AND RR-VALUES INITIALIZED FOR ASCAT
C  FUNCTIONS:
C           ALOS-LINE OF SITE ATTENUATION
C           ASCAT-THE SCATTER ATTENUATION
C
        COMPLEX ZGND
        COMMON /ACR1/DLSA,WN,WAE,DL(2),THE(2),ZGND,HZ(2),HE(2)
        COMMON /LOS/WLS
        COMMON /PREOUT/DLA,EMD,AED,LLOS,LSCAT,THA,XAE
        COMMON /IQLR/DELTAH,HG(5)
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
        COMMON /SCAT/AD,RR,ETQ,HOS
        COMMON /LSFIX/LSFIX
        DATA THIRD/0.333333333333/
C
C  BEGIN
C
        IF (LSFIX.EQ.1) GOTO 40
 10     IF (D.GE.DLSA) GOTO 30
 20     IF (LLOS.EQ.0) GOTO 40
        GOTO 90
 30     IF (LSCAT.EQ.0) GOTO 100
        GOTO 130
C
C  COEFFICIENTS FOR THE LINE-OF-SITE RANGE
C  INITIALIZE 'WLS' BEFORE CALLING ALOS
C
 40     WLS=210./(210.+WN*DELTAH)
        D0=0.5*DLA
        IF (AED.GE.0.) GOTO 60
        D0=AMAX1(D0,AMIN1(-AED/EMD,DLA-2.E03))
        GOTO 70
 60     D0=AMIN1(D0,1.908*WN*HZ(1)*HZ(2))
 70     CONTINUE
        D1=(DLA-D0)*0.25+D0
        D2=DLSA
        A0=ALOS(D0)
        A1=ALOS(D1)
        A2=EMD*D2+AED
        Q=ALOG(D2/D0)
        AK2=((D1-D0)*(A2-A0)-(D2-D0)*(A1-A0))/((D1-D0)*Q-(D2-D0)*
     &       ALOG(D1/D0))
        IF (AK2.LT.0.) AK2=0.
        AK1=(A2-A0-AK2*Q)/(D2-D0)
        IF (AK1.GE.0.) GOTO 80
        AK1=0.
        AK2=(A2-A0)/Q
 80     AEL=A0-AK1*D0-AK2*ALOG(D0)
        LLOS=1
 90     ACR=AK1*D+AK2*ALOG(D)+AEL
        GOTO 160
C
C  COEFFICIENTS FOR THE SCATTER RANGE
C  SET UP INITIAL CONDITIONS FOR ASCAT
C
 100    AD=DL(1)-DL(2)
        RR=HE(2)/HE(1)
        IF (AD.GE.0.) GOTO 105
        AD=-AD
        RR=1./RR
 105    ETQ=(5.67E-06*XN-2.32E-03)*XN+0.031
        HOS=-15.
        D5=DLA+200.E03
        D6=D5+200.E03
        A6=ASCAT(D6)
        A5=ASCAT(D5)
        IF (A5.LT.1000.) GOTO 110
        EMS=EMD
        AES=AED
        DX=10.E06
        GOTO 120
 110    EMS=(A6-A5)/200.E03
        DZ=AMAX1(DLA+0.238*XAE*ALOG(FREQ),(A5-AED-EMS*D5)/(EMD-EMS))
        DX=AMAX1(DZ,DLSA)
        AES=(EMD-EMS)*DX+AED
 120    LSCAT=1
 130    IF (D.LE.DX) GOTO 140
        GOTO 150
 140    ACR=EMD*D+AED
        GOTO 160
 150    ACR=EMS*D+AES
 160    IF (ACR.LT.0) ACR=0.
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        FUNCTION ALOS(D)
C
C  THE LINE-OF-SITE ATTENUATION AT DISTANCE D
C  A CONVEX COMBINATION OF PLANE EARTH FIELDS AND DIFFRACTED FIELDS
C   INPUT:
C         D-DISTANCE
C         DELTAH, TERRAIN PARAMETER CHARACTERIZING MEDIAN PROPAGATION
C                  CONDITIONS FOR THE PROFILE (METERS)
C         HE(1) AND HE(2), THE EFFECTIVE ANTENNA HEIGHTS
C         WN, THE WAVE NUMBER
C         EMD,SLOPE OF THE CURVE OF DIFFRACTION ATTENUATION VERSUS DISTANCE
C         AED, ESTIMATED DIFFRACTION ATTENUATION BELOW FREE SPACE
C         HZ(1) AND HZ(2), THE GROUND WAVE EFFECTIVE HEIGHTS
C         WLS, VALUE INITIALIZED IN ACR
C   OUTPUT:
C         ALOS, THE LINE OF SITE ATTENUATION
C
        COMPLEX ZGND
        COMMON /IQLR/DELTAH,HG(5)
        COMMON /ACR1/DLSA,WN,WAE,DL(2),THE(2),ZGND,HZ(2),HE(2)
        COMMON /PREOUT/DLA,EMD,AED,LLOS,LSCAT,THA,XAE
        COMMON /LOS/WLS
C
C  BEGIN
C
        Q=(1.-0.8*EXP(-D/50.E03))*DELTAH
        S=0.78*Q*EXP(-(Q/16.)**0.25)
        Q=HE(1)+HE(2)
        Q=WN*S*Q/SQRT(D**2+Q**2)
        DZ=EXP(-Q/(1.+Q))
        ALOS=EMD*D+AED
        Q=WN*HZ(1)*HZ(2)*2./D
        A=-4.343*ALOG((DZ-2.*COS(Q))*DZ+1.)-ALOS
        IF (A.LT.0) ALOS=WLS*A+ALOS
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        FUNCTION ASCAT(D)
C
C  THE SCATTER ATTENUATION AT DISTANCE D
C  USES AN APPROXIMATION TO THE METHODS OF NBS TN101 WITH CHECKS
C  FOR INADMISSABLE SITUATIONS
C
C  INPUT:
C        AD,ETQ,HOS,RR-INITIAL CONDITIONS DETERMINED IN ACR
C        AE-EFFECTIVE EARTH'S RADIUS
C        FREQ-FREQUENCY,MHZ
C        HE(1) AND HE(2)-EFFECTIVE ANTENNA HTS
C        THE(1) AND THE(2)-THE HORIZON ELEVATION ANGLES
C        WN-WAVE NUMBER
C        XN-THE SURFACE REFRACTIVITY
C  OUTPUT:
C        ASCAT-THE SCATTER ATTENUATION AT DISTANCE D
C  FUNCTIONS:
C           HOF-THE H0 FUNCTION FOR SCATTER FIELDS
C           AHD-THE F FUNCTION FOR SCATTER FIELDS
C
        COMPLEX ZGND
        COMMON /SCAT/AD,RR,ETQ,HOS
        COMMON /RAD/AE
        COMMON /PREOUT/DLA,EMD,AED,LLOS,LSCAT,THA,XAE
        COMMON /ACR1/DLSA,WN,WAE,DL(2),THE(2),ZGND,HZ(2),HE(2)
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
C
C  BEGIN
C
        IF (HOS.GT.15.) GOTO 10
        TH=D/AE+THE(1)+THE(2)
        R2=2.*WN*TH
        R1=R2*HE(1)
        R2=R2*HE(2)
        IF ((R1.GT.0.2).OR.(R2.GT.0.2)) GOTO 10
        ASCAT=1001.
 10     SS=(D-AD)/(D+AD)
        Q=RR/SS
        IF (SS.LT.0.1) SS=0.1
        IF (Q.LT.0.1) Q=0.1
        IF (Q.GT.10.) Q=10.
        Z0=(D-AD)*(D+AD)*TH*0.25/D
        ETT=(ETQ*EXP(-(Z0/8.0E03)**6)+1.)*Z0/1.7556E03
        ET=ETT
        IF (ETT.LT.1.) ETT=1.
        H0=(HOF(R1,ETT)+HOF(R2,ETT))*0.5
        H0=H0+AMIN1(H0,(1.38-ALOG(ETT))*ALOG(SS)*ALOG(Q)*0.49)
        IF (H0.LT.0.) H0=0.
        IF (ET.LT.1.0) H0=ET*H0+(1.-ET)*4.343*ALOG(((1.+1.4142/R1)*
     &       (1.+1.4142/R2))**2*(R1+R2)/(R1+R2+2.8284))
        IF ((H0.LE.15.0).OR.(HOS.LT.0.)) GOTO 30
 20     H0=HOS
 30     HOS=H0
        TH=THA+D/AE
        ASCAT=AHD(TH*D)+4.343*ALOG(FREQ*TH**4)-0.1*(XN-301.)*
     &       EXP(-TH*D/40.E03)+H0
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        FUNCTION HOF(R,ET)
C
C  THE H0 FUNCTION FOR SCATTER FIELDS
C
        DIMENSION A(5),B(5)
        EQUIVALENCE (Q,IQ)
        DATA A/25.,80.,177.,395.,705./
        DATA B/24.,45.,68.,80.,105./
        IT=ET
        IF (IT.LE.0) GOTO 20
        IF (IT-5) 40,30,20
 10     IT=1
        GOTO 30
 20     IT=5
 30     Q=0.
        GOTO 50
 40     Q=ET-IT
 50     X=(1./R)**2
        HOF=4.343*ALOG((A(IT)*X+B(IT))*X+1.)
        IF (IQ.EQ.0) GOTO 60
        HOF=(1.-Q)*HOF+Q*4.343*ALOG((A(IT+1)*X+B(IT+1))*X+1.)
 60     RETURN
        END
C
C------------------------------------------------------------------------------
C
        FUNCTION AHD(TD)
C
C  THE F(TH*D) FUNCTION FOR SCATTER FIELDS
C
        DIMENSION A(3),B(3),C(3)
        DATA A/133.4,104.6,71.8/
        DATA B/0.332E-03,0.212E-03,0.157E-03/
        DATA C/-4.343,-1.086,2.171/
        I=1
        IF (TD.LE.10.E03) GOTO 10
        I=2
        IF (TD.LE.70.E03) GOTO 10
        I=3
 10     AHD=A(I)+B(I)*TD+C(I)*ALOG(TD)
        RETURN
        END
C
C------------------------------------------------------------------------------
C
        SUBROUTINE EPM73(H,XLOSS)
C
C  IMPLEMENTATION OF MODEL EPM-73, AS DESCRIBED IN:
C   "AN EMPIRICAL PROPAGATION MODEL (EPM-73)"
C    M. N. LUSTGARTEN AND JAMES A. MADISON
C    IEEE TRANSACTIONS ON ELECTROMAGNETIC COMPATIBILITY,
C     VOL. EMC-19, NO. 3, AUGUST 1977
C
C  THIS ADDED 890801, USED IN PLACE OF LONGLEY-RICE MODEL UNDER CERTAIN
C  CONDITIONS; CALLED FROM MODULE ALONG, REFER TO COMMENTS THEREIN
C
        REAL H(*)
C
C  ACCESS TO GLOBAL COMMON BLOCKS WHICH CONTAIN ALL THE PARAMETERS WE NEED
C
        COMMON /CONST/FSM2FT,FM2FT,FKM2SM,FNM2SM,FR2D,PI
        COMMON /GEOM/DST,HST,HSR,HTS,HRS,HTE,HRE,LT,LR,DLT,DLR,
     &       ALPH00,BETA00,THET00
        COMMON /PARAM/FREQ,XN,SIGMA,EPS,HUMID,IPOL,RE,AK,KEPM
        COMMON /PROFIL/NELEV,DINC
C
C  LOCAL DECLARATIONS
C
        REAL LAM,LFS,M,LDC,LD2,LD,LTS
C
C  FIRST, COMPUTE A FIGURE EQUIVALENT TO ANTENNA HEIGHT ABOVE AVERAGE TERRAIN.
C  THIS IS ACTUALLY ACCOMPLISHED BY TAKING THE AVERAGE OF THE DIFFERENCE
C  BETWEEN THE STRAIGHT LINE-OF-SIGHT LINE FROM XMTR TO RCVR AND THE GROUND
C  ELEVATION AT EACH POINT ON THE PROFILE.  THIS COMPENSATES FOR CASES WHERE
C  THE PATH IS CLEARLY LINE-OF-SIGHT AND THE TERRAIN IS FAIRLY SMOOTH, BUT
C  THE GROUND SLOPES STEADILY UPWARD OR DOWNWARD FROM XMTR TO RCVR; IN SUCH
C  CASES, JUST TAKING THE AVERAGE ELEVATION OF THE WHOLE PROFILE LEADS TO
C  INCONSISTENT (AND INCORRECT) RESULTS.
C
        X1=0.
        Y1=H(1)+HST
        X2=DST
        Y2=H(NELEV)+HSR
        ZM=(Y2-Y1)/(X2-X1)
        ZB=Y1-(ZM*X1)
        XTOT=0.
        DO 10 I=1,NELEV
           HTLOS=ZM*(FLOAT(I-1)*DINC)+ZB
           XTOT=XTOT+(HTLOS-H(I))
 10     CONTINUE
        AVET=XTOT/FLOAT(NELEV)
        IF (AVET.LT.0.001) AVET=0.001
C
C  H1  = AVERAGE TRANSMITTER HEIGHT ABOVE TERRAIN IN METERS (SEE ABOVE)
C  H2  = RECEIVER STRUCTURAL HEIGHT (ABOVE GROUND) IN METERS
C  LAM = WAVELENGTH IN METERS
C  D   = DISTANCE FROM TRANSMITTER TO RECEIVER IN KILOMETERS
C  F   = FREQ IN MHZ
C
        H1=AVET/FM2FT
        H2=HSR/FM2FT
        F=FREQ
        LAM=299.791/FREQ
        D=DST/FKM2SM
C
C  COMPUTE FREE-SPACE LOSS AND RADIO LINE-OF-SIGHT DISTANCE FOR BOTH MODELS.
C  ALSO, D2 APPLIES TO BOTH, BUT IS UNDEFINED WHEN F IS LESS THAN 40 MHZ
C
        LFS=33.+20.*LOG10(F)+20.*LOG10(D)
        DLOS=SQRT(17.*H1)+SQRT(17.*H2)
        IF (F.GE.40.) THEN
           IF (F.LE.160.) THEN
              D2=DLOS-48.3*LOG10(F)+163.
           ELSE
              D2=DLOS-16.1*LOG10(F)+91.8
           ENDIF
        ENDIF
C
C  COMPUTE THE LOSS BY THE LOW H/LAMBDA MODEL - THIS DOES NOT APPLY IF
C  FREQUENCY IS GREATER THAN 1000 MHZ OR ANTENNA HEIGHT IS GREATER THAN 300 M
C
        XLO=-1.
        IF ((F.GT.1000.).OR.(H1.GT.300.).OR.(H2.GT.300.)) GOTO 100
C
C  PARAMETER H0 IS THE MINIMUM EFFECTIVE-ANTENNA-HEIGHT PARAMETER, WHICH
C  IS COMPUTED HERE BY THE FORMULA FOR "AVERAGE" LAND CONDITIONS IF WE
C  HAVE VERTICAL POLARIZATION, AND IS 0 IF HORIZONTAL POLARIZATION. AT
C  THE MOMENT, CIRCULAR POLARIZATION IS UNDEFINED
C
        IF (IPOL.EQ.1) THEN
           IF (F.LE.20.) THEN
              H0=10.**(3.61-2.*LOG10(F))
           ELSE
              H0=10.**(2.74-1.33*LOG10(F))
           ENDIF
        ELSE
           H0=0.
        ENDIF
C
C  CONTINUE WITH LOW H/LAMBDA MODEL COMPUTATIONS
C
        H1P=SQRT(H0**2+H1**2)
        H2P=SQRT(H0**2+H2**2)
        DCF=10.**(LOG10(F)+.75*LOG10(H1P*H2P)-3.92)
        IF (F.LE.100.) THEN
           DC=129./SQRT(F)
        ELSE
           DC=59.9/(F**(1./3.))
        ENDIF
        IF ((DCF.GT.DC).OR.(D.LT.DCF)) GOTO 100
        M=MIN(((F**(1./3.))/7.),.5)
        IF (D.LE.DC) THEN
           XLO=111.-15.*LOG10(H1P*H2P)+40.*LOG10(D)
           GOTO 100
        ENDIF
        IF ((D.LE.D2).OR.(F.LT.40.)) THEN
           LDC=111.-15.*LOG10(H1P*H2P)+40.*LOG10(DC)
           XLO=LDC+20.*LOG10(D/DC)+.62*M*(D-DC)
           GOTO 100
        ENDIF
        LD2=111.-15.*LOG10(H1P*H2P)+40.*LOG10(D2)
        XLO=LD2+40.*LOG10(D/D2)
C
C  NOW COMPUTE THE HIGH H/LAMBDA MODEL LOSS. DOES NOT APPLY WHEN F < 40 MHZ
C
 100    XHI=LFS+5.
        IF (F.LT.40.) GOTO 200
        A=(2.08E8*DLOS)/(1000.-3.75*DLOS)
        TMP=H1*H2*F
        IF (TMP.LE.A) THEN
           D1=(1.1*TMP)/3.47E5
        ELSE
           P=.6+TMP*1.08E-8
           D1=1.1*P*DLOS
        ENDIF
        IF (D.LE.D1) GOTO 200
        IF (D.LE.D2) THEN
           LD=(50.*(D-D1))/(D2-D1)
           XHI=LFS+5.+LD
           GOTO 200
        ENDIF
        LTS=20.*LOG10(D/D2)
        XHI=LFS+55.+LTS
C
C  BOTH MODELS COMPUTED; TAKE THE LARGEST OF THE TWO RESULTS. NOTE THAT IF
C  THE LOW MODEL DID NOT APPLY, XLO WILL BE -1, AND THE HIGH MODEL ALWAYS
C  GIVES A VALID LOSS, SO THIS ROUTINE WILL ALWAYS RETURN AN ANSWER
C
 200    XLOSS=MAX(XLO,XHI)
        RETURN
        END
